/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/

/*
 * vario.c: basic variogram model functions (init, print, update, etc.)
 */
#include "defs.h"

#include <stdio.h>
#include <stdlib.h> /* getenv() */
#include <ctype.h> /* toupper() */
#include <float.h> /* DBL_MIN */
#include <math.h>
#include <string.h>

#include "/gstat-2.1.0/meschach/matrix.h"
#include "/gstat-2.1.0/meschach/matrix2.h"

#include "userio.h"
#include "data.h"
#include "utils.h"
#include "debug.h"
#include "vario.h"
#include "vario_fn.h"
#include "glvars.h"
#include "lex.h" /* read_variogram() */
#include "read.h" /* for vario() only */
#include "vario_fn.h"/*KS added*/
#include "gstat_main.h"/*KS added*/

static int is_valid_cs(const VARIOGRAM *aa, const VARIOGRAM *bb,
	const VARIOGRAM *ab);
static int is_posdef(MAT *m);
static void strcat_tm(char *cp, ANIS_TM *tm);
extern char gerr_text[ERR_TEXT_SIZE];

const V_MODEL v_models[] = { /* the variogram model ``data base'': */
	{	 NOT_SP, "Nsp", "Nsp (not specified)",
		"# NSP: should never occur",
		"# NSP: should never occur",
		NULL, NULL },
	{	EXPONENTIAL, "Exp", "Exp (exponential)", 
		"Exp(a,x) = 1 - exp(-x/a)",
		"Exp(a,x) = exp(-x/a)",
		fn_exponential, da_fn_exponential },
	{ 	SPHERICAL, "Sph", "Sph (spherical)", 
		"Sph(a,x) = (x < a ? (1.5 * x/a) - 0.5*((x/a)**3) : 1)",
		"Sph(a,x) = (x < a ? (1 - ((1.5 * x/a) - 0.5*((x/a)**3))) : 0)",
		fn_spherical, da_fn_spherical },
	{ 	CIRCULAR, "Cir", "Cir (circular)", 
		"Cir(a,x) = (x < a ? ((2*x)/(pi*a))*sqrt(1-(x/a)**2)+(2/pi)*asin(x/a) : 1)",
		"Cir(a,x) = (x < a ? (1-(((2*x)/(pi*a))*sqrt(1-(x/a)**2)+(2/pi)*asin(x/a))) : 0)",
		fn_circular, NULL },
	{	LINEAR, "Lin", "Lin (linear)",  /* one-parameter (a = 0), or two-parameter with sill */
		"Lin(a,x) = (a > 0 ? (x < a ? x/a : 1) : x)",
		"Lin(a,x) = (a > 0 ? (x < a ? (1 - x/a) : 0) : 1 - x)",
		fn_linear, da_fn_linear },
	{	GAUSSIAN, "Gau", "Gau (gaussian)", 
		"Gau(a,x) = 1 - exp(-((x/a)**2))",
		"Gau(a,x) = exp(-((x/a)**2))",
		fn_gaussian, da_fn_gaussian },
	{	BESSEL, "Bes", "Bes (bessel)", 
		"# Bessel model not suppurted by gnuplot",
		"# Bessel model not suppurted by gnuplot",
		fn_bessel, NULL },
	{	PENTASPHERICAL, "Pen", "Pen (pentaspherical)", 
		"Pen(a,x) = (x < a ? ((15.0/8.0)*(x/a)+(-5.0/4.0)*((x/a)**3)+\
(3.0/8.0)*((x/a)**5)) : 1)",
		"Pen(a,x) = (x < a ? (1 - ((15.0/8.0)*(x/a)+(-5.0/4.0)*((x/a)**3)+\
(3.0/8.0)*((x/a)**5))) : 0)",
		fn_pentaspherical, da_fn_pentaspherical },
	{	PERIODIC, "Per", "Per (periodic)", 
		"Per(a,x) = 1 - cos(2*pi*x/a)",
		"Per(a,x) = cos(2*pi*x/a)",
		fn_periodic, da_fn_periodic },
	{ 	HOLE, "Hol", "Hol (hole)",
		"Hol(a,x) = 1 - sin(x/a)/(x/a)",
		"Hol(a,x) = sin(x/a)/(x/a)",
		fn_hole, da_fn_hole },
	{	LOGARITHMIC, "Log", "Log (logarithmic)", 
		"Log(a,x) = log(x + a)",
		"Log(a,x) = 1 - log(x + a)",
		fn_logarithmic, da_fn_logarithmic },
	{	POWER, "Pow", "Pow (power)",
		"Pow(a,x) = x ** a",
		"Pow(a,x) = 1 - x ** a",
		fn_power, da_fn_power },
	{	SPLINE, "Spl", "Spl (spline)", 
		/* Wackernagel 2nd ed., p. 225 -- not working yet */
		"Spl(a,x) = x == 0 ? 0 : x * x * log(x)",
		"Spl(a,x) = x == 0 ? 1 : 1 - x * x * log(x)",
		fn_spline, NULL },
	{	NUGGET, "Nug", "Nug (nugget)", 
		"Nug(a,x) = 1 # I don't let gnuplot draw what happens at x=0",
		"Nug(a,x) = 0 # I don't let gnuplot draw what happens at x=0",
		fn_nugget, da_is_zero },
	{	MERROR, "Err", "Err (Measurement error)", 
		"Err(a,x) = 1 # I don't let gnuplot draw what happens at x=0",
		"Err(a,x) = 0 # I don't let gnuplot draw what happens at x=0",
		fn_nugget, da_is_zero },
	/* the folowing two should always be the last ``valid'' one: */
	{	INTERCEPT, "Int", "Int (Intercept)",
		"Int(a,x)   = 1",
		"Int(a,x)   = 1",
		fn_intercept, da_is_zero },
	{	NOT_SP, NULL, NULL, NULL, NULL, NULL, NULL }
};

const char *vgm_type_str[] = { 
	"not specified",
	"semivariogram",
	"cross variogram",
	"covariogram",
	"cross covariogram"
};

VARIOGRAM *init_variogram(VARIOGRAM *v) {
/*
 * initializes one variogram structure
 * if v is NULL, memory is allocated for the structure
 */
	int i;

	if (!v /*== NULL*/)
		v = (VARIOGRAM *) emalloc(sizeof(VARIOGRAM)); /*KS changed from emalloc*/

	v->id = v->id1 = v->id2 = -1;
	v->n_models = 0;
	v->is_valid_covariance = 1;
	v->isotropic = 1;
	v->n_fit = 0;
	v->descr = v->fname = v->fname2 = /*(char *)*/ NULL;
	v->max_range = (double) DBL_MIN;
	v->sum_sills = 0.0;
	v->measurement_error = 0.0;
	v->max_val = 0.0;
	v->min_val = 0.0;
	v->block_semivariance_set = 0;
	v->block_covariance_set = 0;
	v->block_covariance = -1.0;
	v->block_semivariance = -1.0;
	if (NULL == v->part) v->part = (VGM_MODEL *) emalloc(4/*INIT_N_VGMM*/ * sizeof(VGM_MODEL));
	for (i = 0; i < 4/*INIT_N_VGMM*/; i++)
		init_variogram_part(&(v->part[i]));
	v->max_n_models = 4/*INIT_N_VGMM*/;
	v->ev = init_ev();

	return v;
}

void init_variogram_part(VGM_MODEL *p) {
	p->sill = 0.0;  /*KS separated statements*/
    p->range = 0.0;
	p->model = NOT_SP;
	p->fit_sill = 1;   /*KS separated statements*/
    p->fit_range = 1;
	p->fnct = NULL;       /*KS separated statements*/
    p->da_fnct = NULL;
	p->tm_range = NULL;
	p->id = -1;
}

SAMPLE_VGM *init_ev(void) {
	SAMPLE_VGM *ev = NULL;

	ev = (SAMPLE_VGM *) emalloc(sizeof(SAMPLE_VGM));
	set_mv_double(&(ev->cutoff));
	set_mv_double(&(ev->iwidth));
	ev->gamma = NULL;
	ev->dist = NULL;
	ev->nh = NULL;
	ev->pairs = NULL;
	ev->n_max = 0;
	ev->n_est = 0;
	ev->zero = ZERO_DEFAULT;
	ev->plot_numbers = 1;
	ev->is_directional = 0;
	ev->evt = NOTSPECIFIED;
	ev->fit = NO_FIT;
	ev->recalc = 1;
	ev->refit = 1;
	ev->pseudo = 0;
	ev->is_asym = -1;
	ev->map = NULL;
    ev->m_memsize = MAX_EV_SIZE; /*KS added*/
	ev->direction.x = 1.0;   /*KS separated statements*/
	ev->direction.y = 0.0;
    ev->direction.z = 0.0;
	return ev;
}

void free_variogram(VARIOGRAM *v) {
	assert(v /*!= NULL*/);
	if (v->ev) {
		if (v->ev->n_max > 0) {
			efree(v->ev->gamma);
			efree(v->ev->dist);
			efree(v->ev->nh);
			if (v->ev->pairs)
				efree(v->ev->pairs);
		}
		efree(v->ev);
	}
	efree(v->part);
	efree(v);
}

void logprint_variogram(const VARIOGRAM *v, int verbose) {
	printlog("%s", sprint_variogram(v, verbose));
}

/*void fprint_variogram(FILE *f, const VARIOGRAM *v, int verbose) {
	fprintf(f, "%s", sprint_variogram(v, verbose));
} */

const char *sprint_variogram(const VARIOGRAM *v, int verbose) {
/* prints contents of VARIOGRAM v on string */
	static char tmp[1024/*ERROR_BUFFER_SIZE*/] = "", s[1024/*ERROR_BUFFER_SIZE*/] = "";
	int i, j, k;

	if (v->id1 < 0 && v->id2 < 0)
		return s; /* never set */

	if ((!(v->descr) /*== NULL*/ || v->n_models == 0) && (!(v->fname) /*== NULL*/))
		return s; /* nothing to print */

	if (v->id1 == v->id2)
		sprintf(s, "variogram(%s)", name_identifier(v->id1));
	else
		sprintf(s, "variogram(%s,%s)", name_identifier(v->id1),
			name_identifier(v->id2));

	if (v->fname) {
		sprintf(tmp, ": '%s'", v->fname);
		strcat(s, tmp);
	}

	if (v->descr && v->n_models > 0) {
		sprintf(tmp, ": %s", v->descr);
		strcat(s, tmp);
	}

	strcat(s, ";\n");

	if (verbose == 0)
		return s;

	for (i = 0; i < v->n_models; i++) {
		sprintf(tmp, "# model: %d type: %s sill: %g range: %g\n",
			i, v_models[v->part[i].model].name_long,
			v->part[i].sill, v->part[i].range);
		strcat(s, tmp);
		if (v->part[i].tm_range /*!= NULL*/) {
			sprintf(tmp, "# range anisotropy, rotation matrix:\n");
			strcat(s, tmp);
			for (j = 0; j < 3; j++) {
				for (k = 0; k < 3; k++) {
					sprintf(tmp, "%s%8.4f", k == 0 ? "# " : " ",
						v->part[i].tm_range->tm[j][k]);
					strcat(s, tmp);
				}
				strcat(s, "\n");
			}
		}
	}
	sprintf(tmp, "# sum sills %g, max %g, min %g, flat at distance %g\n",
		v->sum_sills, v->max_val, v->min_val, v->max_range);
	strcat(s, tmp);
	return s;
}

void fprint_variogram(CEFile* f, const VARIOGRAM *v)    /*KS replaced*/
/* prints contents of VARIOGRAM v on dest */
{
	int i, j, k;
    char buff[BUFF_SIZE];

	sprintf(buff, "variogram[%d]: %s\n", v->id, v->descr);
    f->fprintf(buff);
	for (i = 0; i < v->n_models; i++)
	{
		sprintf( buff, "model: %d type: %s sill: %g range: %g\n",
			i, v_models[v->part[i].model].name_long,
			v->part[i].sill, v->part[i].range);
        f->fprintf(buff);
		if (v->part[i].tm_range != NULL)
        {
			sprintf( buff, "range anisotropy, rotation matrix:\n");
            f->fprintf(buff);
			for (j = 0; j < 3; j++)
            {
				for (k = 0; k < 3; k++)
                {
					sprintf( buff, " %8.4f", v->part[i].tm_range->tm[j][k]);
                    f->fprintf(buff);
                }
				sprintf( buff, "\n");
                f->fprintf(buff);
			}
		}
	}
	sprintf( buff, "sum sills %g, max_val %g, min_val %g, flat at distance %g\n",
		v->sum_sills, v->max_val, v->min_val, v->max_range);
    f->fprintf(buff);
    f->commit();
	return;
}

void update_variogram(VARIOGRAM *vp) {
/*
 * update min/max, n_fit, descr
 * assumes that models are not changed: they can only be changed through
 * read_variogram();
 */
	char s[LENGTH_OF_MODEL], *cp;
	VGM_MODEL *p;
	int i;

	vp->descr = (char *) erealloc(vp->descr,
		vp->max_n_models * LENGTH_OF_MODEL * sizeof(char));
	cp = vp->descr;
	*cp = '\0';
	/* update sum_sills: */
	vp->sum_sills = vp->min_val = vp->max_val = 0.0;
	vp->n_fit = 0;
	vp->max_range = DBL_MIN;
	for (i = 0; i < vp->n_models; i++) {
		p = &(vp->part[i]);
		vp->sum_sills += p->sill;
		if (p->sill < 0.0)
			vp->min_val += p->sill;
		else
			vp->max_val += p->sill;
		vp->max_range = MAX(p->range, vp->max_range);

		if (p->model == BESSEL || p->model == GAUSSIAN ||
				p->model == EXPONENTIAL || p->model == LOGARITHMIC ||
				p->model == POWER || p->model == PERIODIC ||
				(p->model == LINEAR && p->range == 0)) 
					/* sill is reached asymptotically or oscillates */
			vp->max_range = DBL_MAX;
		else  /* transitive model: */
			vp->max_range = MAX(p->range, vp->max_range);

		if (p->fit_sill == 0)
			strcat(cp, "@ ");
		sprintf(s, gl_format, i == 0 ? p->sill : fabs(p->sill));
		strcat(cp, s);
		strcat(cp, " ");
		sprintf(s, "%s(", v_models[p->model].name);
		strcat(cp, s);
		if ((p->model == LINEAR && p->range == 0.0) || p->model == NUGGET || p->model == INTERCEPT)
			p->fit_range = 0; /* 1 would lead to singularity */
		else if (p->fit_range == 0)
			strcat(cp, "@ ");
		sprintf(s, gl_format, p->range);
		strcat(cp, s);
		if (p->tm_range /*!= NULL*/) 
			strcat_tm(cp, p->tm_range);
		strcat(cp, ")");
		if (i != vp->n_models - 1)
			strcat(cp, vp->part[i+1].sill < 0.0 ? " - " : " + ");
		if (p->model == LOGARITHMIC || p->model == POWER || p->model == INTERCEPT
				|| (p->model == LINEAR && p->range == 0))
		 	vp->is_valid_covariance = 0;
		if (p->fit_sill)
			vp->n_fit++;
		if (p->fit_range)
			vp->n_fit++;
		if (p->model == MERROR)
			vp->measurement_error += p->sill;
	}
	return;
}

static void strcat_tm(char *cp, ANIS_TM *tm) {
	char s[100];

	strcat(cp, ",");
	sprintf(s, gl_format, tm->angle[0]);
	if (TM_IS3D(tm)) {
		strcat(cp, s); strcat(cp, ",");
		sprintf(s, gl_format, tm->angle[1]);
		strcat(cp, s); strcat(cp, ",");
		sprintf(s, gl_format, tm->angle[2]);
	}
	strcat(cp, s); strcat(cp, ",");
	sprintf(s, gl_format, tm->ratio[0]);
	if (TM_IS3D(tm)) {
		strcat(cp, s); strcat(cp, ",");
		sprintf(s, gl_format, tm->ratio[1]);
	}
	strcat(cp, s);
}

double get_max_sill(int n) {
	int i, j;
	VARIOGRAM *vp;
	static double max_sill;

	vp = get_vgm(0);
	max_sill = vp->max_val;
	for (i = 0; i < n; i++) {
		for (j = 0; j <= i; j++) {
			vp = get_vgm(LTI(i,j));
			max_sill = MAX(max_sill, vp->max_val);
		}
	}
	return max_sill;
}

double get_semivariance(const VARIOGRAM *vp, double dx, double dy, double dz) {
/* returns gamma(dx,dy,dz) for variogram v: gamma(h) = cov(0) - cov(h) */

	int i;
	double sv = 0.0, dist = 0.0;

	if (! vp->isotropic) {
		for (i = 0; i < vp->n_models; i++)
			sv += vp->part[i].sill * vp->part[i].fnct(
				transform_norm(vp->part[i].tm_range, dx, dy, dz),
				vp->part[i].range);
	} else {
		dist = transform_norm(NULL, dx, dy, dz);
		if (dist > vp->max_range)
			return vp->sum_sills;
		for (i = 0; i < vp->n_models; i++)
			sv += vp->part[i].sill * vp->part[i].fnct(dist, vp->part[i].range);
	}
	return sv;
}

double get_covariance(const VARIOGRAM *vp, double dx, double dy, double dz) {
/* returns cov(dx,dy,dz) for variogram v */

	int i;
	static int warning = 0;
	double ctmp = 0.0, dist;

	if (! vp->is_valid_covariance && !warning) {
        sprintf(gerr_text,"%s: non-transitive variogram model not allowed as covariance",
			vp->descr);
        pr_warning(gerr_text);
		//pr_warning(
		//	"%s: non-transitive variogram model not allowed as covariance",
		//	vp->descr);
		warning = 1;
	}
	if (! vp->isotropic) {
		for (i = 0; i < vp->n_models; i++)
			ctmp += vp->part[i].sill * (1.0 - vp->part[i].fnct(
				transform_norm(vp->part[i].tm_range, dx, dy, dz),
				vp->part[i].range));
	} else {
		dist = transform_norm(NULL, dx, dy, dz);
		for (i = 0; i < vp->n_models; i++)
			ctmp += vp->part[i].sill * (1.0 - vp->part[i].fnct(dist,
				vp->part[i].range));
	}
	return ctmp;
}

static int is_valid_cs(const VARIOGRAM *aa, const VARIOGRAM *bb,
	const VARIOGRAM *ab)
/*
 * Purpose       : Check Cauchy-Schwartz inequality on cross/variograms    
 * Created by    : Edzer J. Pebesma                                       
 * Date          : may 6th 1992                                          
 * Prerequisites :                                                     
 * Returns       : return nonzero if |g_ab(h)| > sqrt(g_aa(h)g_bb(h))
 * Side effects  : none                                             
 */
{
	int i, check_failed = 0;
	double maxrange = 0, dist, dx, dy, dz;

	for (i = 0; i < aa->n_models; i++)
		if (aa->part[i].range > maxrange)
			maxrange = aa->part[i].range;
	for (i = 0; i < ab->n_models; i++)
		if (ab->part[i].range > maxrange)
			maxrange = ab->part[i].range;
	for (i = 0; i < bb->n_models; i++)
		if (bb->part[i].range > maxrange)
			maxrange = bb->part[i].range;
	for (i = 0; i < 101 && !check_failed; i++) {
		dist = (i * maxrange)/100;
		dx = dy = dz = 0.0;
		if (i % 3 == 0) dx = dist;
		if (i % 3 == 1) dy = dist;
		if (i % 3 == 2) dz = dist;
		if (fabs(get_semivariance(ab, dx, dy, dz)) >
			sqrt(get_semivariance(aa, dx, dy, dz) * 
			get_semivariance(bb, dx, dy, dz))) {
			check_failed = 1; /* yes, the check failed */
			/*pr_warning*//*sprintf(gerr_text,"%s %d %s %d %s %d\n%s\n%s\n%s\n%s\n%s %g %g %g",
				"Cauchy-Schwartz violation: variogram",
				aa->id,",",bb->id, "and cross variogram", ab->id,
				"descriptors: ", aa->descr, bb->descr, ab->descr,
				"first failure on dx, dy and dz:", dx, dy, dz);
            pr_warning(gerr_text); *//*KS removed*/
		}
	} /* for */
	if (check_failed)
		return 0;
	else
		return 1;
}

void check_variography(const VARIOGRAM **v, int n_vars)
/*
 * check for intrinsic correlation, linear model of coregionalisation
 * or else (with warning) Cauchy Swartz
 */
{
	int i, j, k, ic = 0, lmc, posdef = 1;
	MAT **a = NULL;
	double b;

	if (n_vars <= 1)
		return;
/* 
 * find out if lmc (linear model of coregionalization) hold: 
 * all models must have equal base models (sequence and range)
 */
	for (i = 1, lmc = 1; lmc && i < get_n_vgms(); i++) {
		if (v[0]->n_models != v[i]->n_models) 
			lmc = 0;
		for (k = 0; lmc && k < v[0]->n_models; k++)
			if (v[0]->part[k].model != v[i]->part[k].model ||
					v[0]->part[k].range != v[i]->part[k].range)
				lmc = 0;
		for (k = 0; lmc && k < v[0]->n_models; k++)
			if (v[0]->part[k].tm_range /*!= NULL*/) {
				if (!(v[i]->part[k].tm_range) /*== NULL*/)
					lmc = 0;
				else if (
		v[0]->part[k].tm_range->ratio[0] != v[i]->part[k].tm_range->ratio[0] ||
		v[0]->part[k].tm_range->ratio[1] != v[i]->part[k].tm_range->ratio[1] ||
		v[0]->part[k].tm_range->angle[0] != v[i]->part[k].tm_range->angle[0] ||
		v[0]->part[k].tm_range->angle[1] != v[i]->part[k].tm_range->angle[1] ||
		v[0]->part[k].tm_range->angle[2] != v[i]->part[k].tm_range->angle[2]
				)
				lmc = 0;
			} else if (v[i]->part[k].tm_range /*!= NULL*/)
				lmc = 0;
	}
	if (lmc) {
/*
 * check for ic:
 */
		a = (MAT **) emalloc(v[0]->n_models * sizeof(MAT *));
		for (k = 0; k < v[0]->n_models; k++)
			a[k] = m_get(n_vars, n_vars);
		for (i = 0; i < n_vars; i++) {
			for (j = 0; j < n_vars; j++) { /* for all variogram triplets: */
				for (k = 0; k < v[0]->n_models; k++)
					a[k]->me[i][j] = v[LTI(i,j)]->part[k].sill;
			}
		}
		/* for ic: a's must be scaled versions of each other: */
		ic = 1;
		for (k = 1, ic = 1; ic && k < v[0]->n_models; k++) {
			b = a[0]->me[0][0]/a[k]->me[0][0];
			for (i = 0; ic && i < n_vars; i++)
				for (j = 0; ic && j < n_vars; j++)
					if (fabs(a[0]->me[i][j] / a[k]->me[i][j] - b) > EPSILON)
						ic = 0;	
		}
		/* check posdef matrices */
		for (i = 0, lmc = 1, posdef = 1; i < v[0]->n_models; i++) {
			posdef = is_posdef(a[i]);
			if (posdef == 0)
				ic = lmc = 0;
			printlog("%s%s %d\n", posdef == 0 ? "non-" : "",
				"positive definite coefficient matrix in structure", i+1);
		}
		for (k = 0; k < v[0]->n_models; k++)
			efree(a[k]);
		efree(a);

		if (ic) {
			printlog("Intrinsic Correlation found. Good.\n");
			return;
		} else if (lmc) {
			printlog("Linear Model of Coregionalization found. Good.\n");
			return;
		}
	}
/*
 * lmc does not hold: check on Cauchy Swartz
 */
	pr_warning("No Intrinsic Correlation or Linear Model of Coregionalization found");
	if (gl_nocheck == 0) {
//		pr_warning("[add `set nocheck = 1;' to the command file to ignore the following error]\n");
        /*KS changed for Idrisi32*/
        ErrMsg(ER_IMPOSVAL, " Uncheck the 'check correlation' option to ignore the following error: variograms do not satisfy a legal model\n");
//		ErrMsg(ER_IMPOSVAL, "variograms do not satisfy a legal model");
	}
	printlog("Now checking for Cauchy-Schwartz inequalities:\n");
	for (i = 0; i < n_vars; i++)
		for (j = 0; j < i; j++)
			if (is_valid_cs(v[LTI(i,i)], v[LTI(j,j)], v[LTI(i,j)])) {
				printlog("variogram(%s,%s) passed Cauchy-Schwartz\n",
					name_identifier(j), name_identifier(i));
			} else {
				/*pr_warning*/sprintf(gerr_text,"Cauchy-Schwartz inequality found for variogram(%s,%s)",
						name_identifier(j), name_identifier(i) );
                  pr_warning(gerr_text);
                  }      
	return;
}

/* from meschach matrix library (c) , see matrix[2].h
 try CHfactor -- Cholesky L.L' factorisation of A in-situ */
static int is_posdef(MAT *A) {
	/*u_int*/unsigned int	i, j, k;
	Real	sum, tmp;

	for (k = 0; k < A->n; k++)
	{	
		/* do diagonal element */
		sum = A->me[k][k];
		for (j = 0; j < k; j++)
		{
			tmp = A->me[k][j];
			sum -= tmp*tmp;
		}
		if (sum <= 0.0)
			return 0;
		A->me[k][k] = sqrt(sum);

		/* set values of column k */
		for (i = k + 1; i < A->n; i++)
		{
			sum = A->me[i][k];
			sum -= __ip__(A->me[i],A->me[k],(int)k);
			A->me[j][i] = A->me[i][j] = sum/A->me[k][k];
		}
	}
	return 1;
}

double transform_norm(const ANIS_TM *tm, double dx, double dy, double dz) {
/* returns variogram distance given dx, dy, dz and VARIOGRAM v */

	double dist = 0.0, tmp;
	int i;

	if (dx == 0.0 && dy == 0.0 && dz == 0.0)
		return 0.0;
	if (tm /*!= NULL*/) {
		for (i = 0, tmp = 0.0; i < 3; i++) {
			tmp = tm->tm[i][0] * dx + tm->tm[i][1] * dy + tm->tm[i][2] * dz;
			dist += tmp * tmp;
		}
		return sqrt(dist);
	} 
	return sqrt((dx * dx) + (dy * dy) + (dz * dz));
}

double da_general(VGM_MODEL *part, double h) {
	double low, high, range;

	range = MAX(1e-20, part->range);
	low = part->fnct(h, range * (1.0 + DA_DELTA));
	high = part->fnct(h, range * (1.0 - DA_DELTA));
	return part->sill * (low - high) / (2.0 * range * DA_DELTA);
}

int push_variogram_model(VARIOGRAM *v, VGM_MODEL part) {
	int i, max_id, where = -1;
/*
 * add the part submodel to v (if part.id < 0) or else
 * modify the appropriate part of v, having the id of part.id.
 * do a lot of checks, and set .fn and .da_fn functions.
 */

	if (v->n_models == v->max_n_models) {
		v->max_n_models *= 2;
		v->part = (VGM_MODEL *) 
			erealloc(v->part, v->max_n_models * sizeof(VGM_MODEL));
	}
	/*
	 * check some things: 
	 */
	if (part.model == NOT_SP)
		ErrMsg(ER_IMPOSVAL, "model NSP not allowed in variogram structure");
	if (part.range < 0.0)
		ErrMsg(ER_RANGE, "variogram range cannot be negative");
	if (part.model == LINEAR) {
		if (part.range == 0.0)
			part.fit_range = 0;
	} else if (part.model == NUGGET || part.model == INTERCEPT || 
			part.model == MERROR) {
		part.fit_range = 0;
		if (part.range > 0.0) 
			ErrMsg(ER_RANGE, "range must be zero");
	} else if (part.range == 0.0) 
		ErrMsg(ER_RANGE, "range must be positive");
	if (part.model == POWER && part.range > 2.0)
		ErrMsg(ER_RANGE, "power model can not exceed 2.0");

	if (part.id < 0) {
		where = v->n_models;
		v->n_models++;
		for (i = max_id = 0; i < v->n_models; i++)
			max_id = MAX(v->part[i].id, max_id);
		part.id = max_id + 1;
	} else { /* search in list: */
		for (i = 0; where < 0 && i < v->n_models; i++)
			if (v->part[i].id == part.id)
				where = i;
		assert(where >= 0); /* i.e., it should really be in the list */
	}

	if (v->isotropic)
		v->isotropic = (part.tm_range == NULL);  /*KS ?*/

	/* 
	 * check that the .fn and .da_fn functions in v_models 
	 * will indeed be the correct ones: 
	 */
	assert(part.model == v_models[part.model].model);

	v->part[where] = part;
	v->part[where].fnct = v_models[part.model].fn;
	v->part[where].da_fnct = v_models[part.model].da_fn;

	return part.id;
}

VGM_MODEL_TYPE which_variogram_model(const char *m) {
	char s[4];
	int i;

	strncpy(s, m, 3);
	s[0] = toupper(s[0]);
	s[1] = tolower(s[1]);
	s[2] = tolower(s[2]);
	s[3] = '\0';
	for (i = 1; NULL != v_models[i].name; i++)
		if (almost_equals(s, v_models[i].name))
			return v_models[i].model;
	return NOT_SP;
}

double relative_nugget(VARIOGRAM *v) {
	int i;
	double nug = 0.0, sill = 0.0;
	
	assert(v->n_models != 0);

	if (v->n_models == 1)
		return (v->part[0].model == NUGGET ? 1.0 : 0.0);

	for (i = 0; i < v->n_models; i++) {
		if (v->part[i].model == NUGGET)
			nug += v->part[i].sill;
		else
			sill += v->part[i].sill;
	}
	assert(nug + sill > 0.0);
	return (nug/(nug+sill));
}

int vario(int argc, char **argv) {
/* model from to nsteps */
	double dist, from, to;
	int i, is_vgm, nsteps = 0;
	VARIOGRAM vgm;

	is_vgm = almost_equals(argv[0], "se$mivariance");
	if (argc < 3) {
		printlog("usage: %s variogram_model dist [to_dist [n_intervals]]\n", argv[0]);
		exit(0);
	}
	init_variogram(&vgm);
	vgm.id = 0;
	if (read_variogramo(&vgm, string_dup(argv[1])))
		ErrMsg(ER_SYNTAX, argv[1]);
	if (read_double(argv[2], &from))
		ErrMsg(ER_RDFLT, argv[2]);
	if (argc >= 4) {
		if (read_double(argv[3], &to))
			ErrMsg(ER_RDFLT, argv[3]);
		nsteps = 1;
	} else
		to = from;
	if (argc >= 5)
		if (read_int(argv[4], &nsteps))
			ErrMsg(ER_RDINT, argv[4]);
	if (DEBUG_DUMP)
		logprint_variogram(&vgm, 1);
	if (nsteps < 0)
		ErrMsg(ER_RANGE, "n_steps must be >= 0");
	dist = from;
	for (i = 0; i <= nsteps; i++) {
		printlog("%g %g\n", dist, (is_vgm ? 
			get_semivariance(&vgm, dist, 0, 0) : 
			get_covariance(&vgm, dist, 0, 0)));
		if (i < nsteps) /* nsteps > 0 */
			dist += (to - from)/(1.0*nsteps);
	}	
	return 0;
}
