/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/

/*
 * userio.c: i/o routines for error, warning, log and progress messages
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "defs.h"

/*#ifdef PCRCALC        /*KS*/
/*# define efclose fclose
#else */
# include "/gstat-2.1.0/meschach/err.h"
/*#endif*/

#include "debug.h"
#include "utils.h"
#include "version.h"
#include "userio.h"
#include "gstat_main.h"

#define is_openf(f) (f /*!= NULL*/ && f != stdout && f != stderr)

static FILE *logfile = NULL;

extern int handle;
extern char *directory;
extern unsigned long perc1;
extern unsigned  perc2;

static struct {
	void (*warning_handler)(const char *mess);
	void (*error_handler)(const char *mess, int level);
	void (*printlog_handler)(const char *mess);
	void (*progress_handler)(int this2, int total);
} gstat_handler = { NULL, NULL, NULL, NULL };

static STRING_BUFFER
	*error_prefix = NULL, 
	*error_message = NULL, 
	*warning_message = NULL;

static enum Gstat_errno gstat_errno;

static char *error_messages[] = {
/* 0 */		" %s",
/* 1 */		" bug in function `%s'",
/* 2 */		" variable not set: %s",
/* 3 */		" variable outside valid range: %s",
/* 4 */		" value not allowed : %s",     //KS changed "value not allowed for: %s",
/* 5 */		" no filename set %s",
/* 6 */		" write failed on file `%s'",
/* 7 */		" read failed on file `%s'",
/* 9 */		" cannot read real value from `%s'",
/* 9 */		" cannot read integer from `%s'",
/* 10 */	" syntax error: %s",
/* 11 */	" illegal option or missing argument on `%s'",
/* 12 */	" domain (math) error on `%s'",
/* 13 */	" out of dynamic memory %s",
/* 14 */	" i/o error: %s",
/* 15 */	" no command file%s",
/* 16 */	" %s user interface not compiled in this version",
/* 17 */	" writing to pipe `%s' failed",
/* 18 */	" reading from pipe `%s' failed",
/* 19 */    " function call prevented by secure mode%s",
/* 20 */	" matrix library error: %s"
};

#define MAX_ERRNO (sizeof(error_messages)/sizeof(char *) - 1)

void init_userio(int use_stdio) {
	if (use_stdio) {
		set_gstat_log_file(stdout);
		set_gstat_warning_handler(default_warning);
		set_gstat_error_handler(default_error);
		set_gstat_log_handler(default_printlog);
		set_gstat_progress_handler(default_progress);
	} else {
		/* ... */
	}
	error_prefix    = resize_strbuf(error_prefix, 1024/*ERROR_BUFFER_SIZE*/);
	error_message   = resize_strbuf(error_message, 1024/*ERROR_BUFFER_SIZE*/);
	warning_message = resize_strbuf(warning_message, 1024/*ERROR_BUFFER_SIZE*/);
}

/*
 * error handling function -- print message and error to string, and
 * call error message handler.
 */
/*void gstat_error(char *fname, int line,
	enum Gstat_errno err_nr, const char *msg) {
	char s[30], *buf;
	int len;

	assert(err_nr <= MAX_ERRNO);
	gstat_errno = err_nr;

	if (error_prefix->str[0] != '\0')
		save_strcat(error_message, error_prefix->str);

	save_strcat(error_message, "gstat: ");
	len = strlen(error_message->str);
	buf = error_message->str + len;

#ifdef HAVE_ASPRINTF
	asprintf(&buf, error_messages[err_nr], NULS(msg));
	save_strcat(error_message, buf);
	free(buf);
#else
	sprintf(buf, error_messages[err_nr], save_string(msg));
#endif

	if (DEBUG_DUMP || err_nr == ER_NULL) {
		save_strcat(error_message, " (");
		save_strcat(error_message, fname);
		sprintf(s, ", line %d)", line);
		save_strcat(error_message, s);
	}

	if (err_nr == ER_NULL) {
		save_strcat(error_message, VERSION);
		save_strcat(error_message, GSTAT_OS);
		save_strcat(error_message,
			"\nThis is a bug. Please send the above information, along with\n");
		save_strcat(error_message,
			"the information necessary to reproduce this bug to ");
		save_strcat(error_message, GSTAT_EMAIL);
	}

	gstat_handler.error_handler(error_message->str, err_nr);
	return;
} *//*KS replaced below*/

/* wrapper function for ErrClo(optopt), in case of error command line option */
void gstat_clo_error(char *f, int l, int err, int a) {
	static char s[2];
	sprintf(s, "%c", a);
	gstat_error(f, l, err, s);
} 

/* message() calls for messages preceding a call to ErrMsg() */
void message(char *fmt, ...) {
	va_list args;
	char *buf = NULL;

#ifdef HAVE_VASPRINTF
	va_start(args, fmt);
	vasprintf(&buf, fmt, args);
	va_end(args);
	save_strcat(error_prefix, buf);
	free(buf);
#else
	va_start(args, fmt);
	vsprintf(error_prefix->str, fmt, args);
	va_end(args);
	buf = NULL;
#endif
}

/* print a warning message to string, and call warning message handler */
/*void pr_warning(char *fmt, ...) {
	va_list args;
	char *buf = NULL;

	if (warning_message->max_length < 11)
		resize_strbuf(warning_message, 11);

	warning_message->str[0] = '\0';
	save_strcat(warning_message, "Warning: ");

#ifdef HAVE_VASPRINTF
	va_start(args, fmt);
	vasprintf(&buf, fmt, args);
	va_end(args);
	save_strcat(warning_message, buf);
	free(buf);
#else
	buf = warning_message->str + 9;
	va_start(args, fmt);
	vsprintf(buf, fmt, args);
	va_end(args);
#endif

	gstat_handler.warning_handler(warning_message->str);
} *//*KS replaced below*/

/*void print_progress(int current, int total) {
	gstat_handler.progress_handler(current, total);
}*//*KS uses percent_done*/

/* get the value of gstat errno */
enum Gstat_errno get_gstat_errno(void) {
	return gstat_errno;
}

/* set the internal gstat errno to NO_ERROR, and reset error mesages */
void reset_gstat_errno(void) {
	assert(error_prefix);
	assert(error_message);

	gstat_errno = ER_NOERROR;
	error_prefix->str[0] = '\0';
	error_message->str[0] = '\0';
}

void set_gstat_warning_handler(void (*warning_fn)(const char *message)) {
	gstat_handler.warning_handler = warning_fn;
}

void set_gstat_error_handler(void (*error_fn)(const char *message, int level)) {
	gstat_handler.error_handler = error_fn;
}

void set_gstat_log_handler(void (*logprint)(const char *str)) {
	gstat_handler.printlog_handler = logprint;
}

void set_gstat_progress_handler(void (*progress)(int this2, int total)) {
	gstat_handler.progress_handler = progress;
}

const char *get_gstat_error_message(void) {
	return (const char *) error_message;
}

void default_warning(const char *mess) {

	if (is_openf(logfile))
		printlog("%s\n", mess);

	fprintf(stderr, "%s\n", mess);
	return;
}

void default_error(const char *mess, int level) {

	if (is_openf(logfile))
		printlog("%s\n", mess);

	fprintf(stderr, "%s\n", mess);
	exit(level == 0 ? -1 : level);
}

void printlog(const char *fmt, ...) {
	static STRING_BUFFER *s = NULL;
	va_list args;
	char *buf = NULL;

	if (!s /*== NULL*/)
		s = resize_strbuf(s, 1024/*ERROR_BUFFER_SIZE*/);
	s->str[0] = '\0';

#ifdef HAVE_VASPRINTF
	va_start(args, fmt);
	vasprintf(&buf, fmt, args);
	va_end(args);
	save_strcat(s, buf);
	free(buf);
#else
	va_start(args, fmt);
	vsprintf(s->str, fmt, args);
	va_end(args);
	buf = NULL;
#endif

	gstat_handler.printlog_handler(s->str);
}

void default_printlog(const char *mess) {

	if (DEBUG_SILENT)
		return;

	if (is_openf(logfile))
		fprintf(logfile, "%s", mess);
	else
		fprintf(stdout, "%s", mess);
}

int set_gstat_log_file(FILE *f) {
	int retval;

	if (NULL == f) {
		if (is_openf(logfile)) {
			retval = efclose(logfile);
			logfile = NULL;
			return retval;
		} else {
			logfile = NULL;
			return 1;
		}
	} else
		logfile = f;
	return 0;
}

void default_progress(int current, int total) {
	static int perc_last = -1, sec_last = -1;
	int perc, sec;
	static time_t start;

	if (total <= 0 || DEBUG_SILENT)
		return;

	if (sec_last == -1) {
		start = time(NULL);
		sec_last = 0;
	}
	/* perc = floor(100.0 * (current + 1.0)/(total * 1.0) + 0.5); */
	perc = floor(100.0 * current / total);
	if (perc != perc_last) { /* another percentage -> calculate time: */
		if (perc == 100) { /* reset: */
			fprintf(stderr, "\r%3d%% done\n", perc);
			perc_last = sec_last = -1;
		} else {
			sec = difftime(time(NULL), start);
			if (sec != sec_last) { /* another second -- don't print too often */
				fprintf(stderr, "\r%3d%% done", perc);
				perc_last = perc;
				sec_last = sec;
			}
		}
	}
}

#ifndef PCRCALC
/**************************** meschach error functions ****************/

#define SING_ERR \
"The most frequent occasion for this error is when two (or more) data\n\
points have identical locations. The way to solve it is to add the word\n\
`average' (without quotes) to the data definition. If this does not\n\
resolve the problem, it may be possible that nearly identical data point\n\
locations are the cause. Remove them manually, or set the default value of\n\
the variable `zero' to a larger value (but check the manual for a suitable\n\
value and the side effects this may have!)"

#define MEM_ERR \
"In case you are trying to do global kriging (i.e., no neighbourhood\n\
parameters like `radius' or `max' were specified) with a large data set,\n\
reduce the neighbourhood size and use local kriging. In case you are\n\
fitting a variogram model to a large data set with REML, try another\n\
fitting method."

#define FP_ERR \
"This error may arise from using _very_ large or very small values for\n\
data values, variograms or coordinates. Try to rescale them to a\n\
reasonable range."

void setup_meschach_error_handler(void) {
 	int code;
 	char *err, *hint = "", buf[100];

 	/* set up meschach error handler: */
 	if ((code = setjmp(restart)) == 0)
 		set_err_flag(EF_JUMP); /* make meschach jump on errors */
 	else {
 		/* setjmp() returned non-zero, so we returned from a longjmp(): */
 		switch (code) {
 			case E_MEM:  /* run out of memory */
 				err = "virtual memory exhausted";
 				hint = MEM_ERR;
 				break;
 			case E_SING:  /* singular matrix occurred */
 				err = "singular matrix";
 				hint = SING_ERR;
 				break;
 			case E_POSDEF: /* non-positive definite matrix */
 				err = "non-positive definite matrix";
 				hint = "";
 				break;
 			case E_SIGNAL: /* floating point signal */
 				err = "floating point exception";
 				hint = FP_ERR;
 				break;
 			default:
 				sprintf(buf, "unknown, error code %d", code);
 				err = buf;
 				hint = "";
 				break;
 		}
		printlog("\ngstat caught an error that occurred in the matrix library,\n");
 		printlog("the reason for it was: %s\n\n", err);
		if (*hint)
			printlog("HINT: %s\n\n", hint);
 		ErrMsg(ER_MESCHACH, err);
 	}
}
/*----------------------------------------------------------------------------------------------------------*/
/*KS added older versions*/
void pr_warning(char *fmt, ...) {
	char *warning, *warn_text;
FILE *tmpf;
char *namew;

    namew=(char *) (emalloc(strlen(directory) + 11));
    strcpy(namew,directory);
  	strcat(namew, "gsttmp.err");
    warning=(char *) (emalloc(20));
	strcpy(warning,"W=GSTAT Warning : ");/*1/19/99*/
    tmpf = efopen(namew,"at");/*1/20/99*/
    fprintf(tmpf, warning);
    fprintf(tmpf, fmt);
    fprintf(tmpf, "\n");
    fclose(tmpf);
    efree(warning); efree(namew);
    percent_done(perc1,perc2,handle);
    }

/*KS added Meschach Library error messaging through mes_warning*/
void mes_warning(char *fmt, ...) {
	char *warning, *warn_text;
FILE *tmpf;
char *namew;

    namew=(char *) (emalloc(strlen(directory) + 11));
    strcpy(namew,directory);
  	strcat(namew, "gsttmp.err");
    warning=(char *) (emalloc(strlen(fmt) + 40));
	strcpy(warning,"M=Meschach Library Error (via GSTAT): ");/*1/19/99*/
   	warn_text=(char *) emalloc(strlen(fmt));
    strcpy(warn_text,fmt);
   	strcat(warning,warn_text);
    tmpf = efopen(namew,"at");/*1/20/99*/
    fprintf(tmpf, warning);
    fprintf(tmpf, "\n");
    fclose(tmpf);
    efree(warning); efree(namew); efree(warn_text);
    }

    /*KS Idrisi version*/
void gstat_error(const char *fname, int line, int err_nr, const char *msg) {/*KS changed*/
FILE *tmpf2 = NULL;
char *warning,*tname = NULL;

	if (err_nr < 0 || err_nr > MAX_ERRNO)
		err_nr = 0;

    tname=(char *) (emalloc(strlen(directory) + 11));
    strcpy(tname,directory);
  	strcat(tname, "gsttmp.err");
    warning=(char *) (emalloc(18));
	strcpy(warning,"E=GSTAT Error : ");

    tmpf2 = efopen(tname,"at");
    fprintf(tmpf2, warning);

    fprintf(tmpf2, error_messages[err_nr], NULS(msg));
	if (DEBUG_DUMP)
		fprintf(tmpf2, " (%s, line %d)", fname, line);
/*	if (err_nr == ER_NULL) {
		fprintf(tmpf2, "\nthis is a bug, please send the next line to %s", EMAIL);
		fprintf(tmpf2, "\n%s, line %d, %s %s\n", fname, line, VERSION, GSTAT_OS);
	}*/
	fprintf(tmpf2, "\n");
    fclose(tmpf2);
    efree(warning); efree(tname);
    perc1 = 99; perc2 = 100;
    percent_done(perc1,perc2,handle); /*KS added*//*1/4/99*/  /*/999*/
	exit(err_nr == 0 ? -1 : err_nr);
	return;
}
#endif
