/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/

/*
 * sem.c: calculate sample (cross, co-) variogram from data
 * K.M. refers to changes by Konstantin Malakhanov, see mapio.c
 */
 #include <vcl.h>  /*KS added*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "defs.h"
#include "read.h"
#include "mapio.h"
#include "userio.h"
#include "data.h"
#include "utils.h"
#include "debug.h"
#include "vario.h"
#include "glvars.h"
#include "select.h"
#include "gls.h"
#include "lm.h"
#include "defaults.h"
#include "direct.h"
#include "version.h"
#include "sem.h"

#define SEM_INCREMENT 1000

static double valid_distance(DPOINT *a, DPOINT *b, double max, 
		int symmetric, DATA *d1, DATA *d2, GRIDMAP *m);
static void divide(SAMPLE_VGM *ev);
static SAMPLE_VGM *alloc_exp_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev);
static void fprint_sample_vgm(CEFile* f, const SAMPLE_VGM *ev, int n_list1, int n_list2);      /*KS added*/
static void fprint_header_vgm(CEFile* f, const DATA *d1, const DATA *d2,const SAMPLE_VGM *ev); /*KS added*/
void Pagefprint_sample_vgm(CEFile* f, const SAMPLE_VGM *ev, int n_list1, int n_list2);         /*KS added*/
SAMPLE_VGM * SetEVnest(DATA *a, DATA *b, SAMPLE_VGM *ev);                                      /*KS added*/
SAMPLE_VGM * AllocVGM(DATA *a, DATA *b, SAMPLE_VGM *ev);                                       /*KS added*/
void ResetEVspace( SAMPLE_VGM *ev );
void PageDivide(SAMPLE_VGM *ev);
/* variograms: */                                                        /*KS replaced routines*/
/*static SAMPLE_VGM *semivariogram(DATA *a, SAMPLE_VGM *ev);*/
SAMPLE_VGM * UpFrontSemivariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2);   	/*KS added*/
SAMPLE_VGM * PageSemivariogram(DATA *d, SAMPLE_VGM *ev, CEFile* f2);     	/*KS added*/
SAMPLE_VGM * c_Semivariogram(DATA *d,DATA *b, SAMPLE_VGM *ev,CEFile* f); 	/*KS added*/
/*static SAMPLE_VGM *cross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev);*/
SAMPLE_VGM * UpFrontCross_covariogram(DATA *d, DATA *b,SAMPLE_VGM *ev,CEFile* f2);	/*KS added*/
SAMPLE_VGM * PageCross_covariogram(DATA *a, DATA *b,SAMPLE_VGM *ev,CEFile* f2);   	/*KS added*/
SAMPLE_VGM * c_Cross_covariogram(DATA *d, DATA *b,SAMPLE_VGM *ev,CEFile* f);      	/*KS added*/
/* covariograms: */
/*static SAMPLE_VGM *covariogram(DATA *a, SAMPLE_VGM *ev);*/
SAMPLE_VGM *UpFrontCovariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2);         /*KS added*/
SAMPLE_VGM *PageCovariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2);            /*KS added*/
SAMPLE_VGM *c_Covariogram(DATA *d,DATA *b, SAMPLE_VGM *ev,CEFile* f);       /*KS added*/

/*static SAMPLE_VGM *cross_covariogram(DATA *a, DATA *b, SAMPLE_VGM *ev);*/
SAMPLE_VGM * UpFrontCross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f2);  /*KS added*/
SAMPLE_VGM *PageCross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f2);      /*KS added*/
SAMPLE_VGM *c_Cross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f);         /*KS added*/
static int GetIndex(DATA *d, SAMPLE_VGM *ev);                                      /*KS added*/
static int GeMaxCross_variogramIndex(DATA *a, DATA *b, SAMPLE_VGM *ev);            /*KS added*/
static int GetMaxCross_covariogramIndex(DATA *a, DATA *b,SAMPLE_VGM *ev);          /*KS added*/
static int GetMax_CovariogramIndex(DATA *d, SAMPLE_VGM *ev);                       /*KS added*/

static int get_index(double dist, float iwidth /*SAMPLE_VGM *ev*/);
static void ev2map(VARIOGRAM *v);
static SAMPLE_VGM *load_ev(SAMPLE_VGM *ev, const char *fname);
static SAMPLE_VGM *semivariogram_list(DATA *d, SAMPLE_VGM *ev);
static SAMPLE_VGM *semivariogram_grid(DATA *d, SAMPLE_VGM *ev);
static void push_to_cloud(SAMPLE_VGM *ev, double gamma, double dist,
	unsigned long index);
static void resize_ev(SAMPLE_VGM *ev, unsigned int size);
static void *register_pairs(void *p, unsigned long nh, DPOINT *a, DPOINT *b);

extern int handle;
extern char *directory;
extern bool CORRELOGRAM;
extern int resid;
extern char gerr_text[ERR_TEXT_SIZE];
/*
 * gl_cressie: use Cressie's sqrt(absdiff) estimator;
 * ev->zero:
 *   case ZERO_INCLUDE: use zero distances in first interval (omit);
 *   case ZERO_AVOID:   avoid zero distances;
 *   case ZERO_SPECIAL: make special estimate for distance zero;
 */

/*
 * calculate sample variogram from data
 * calculates variogram, crossvariogram, covariogram or crosscovariogram
 * from sample data. Data are obtained from the central data base (glvars)
 * using get_data(), and the variogram requested is that of data id
 * v->id1 and v->id2 -- a direct (co) variogram when id1 == id2, a cross
 * (co) variogram when id1 != id2. 
 * 
 * if v->fname is set and (one of) id1 or id2 is a dummy data, the
 * actual sample variogram is not calculated but rather read from the
 * file v->fname. This is done to enable separate sample variogram
 * calculation (in batch or on a fast remote computer) and model fitting
 * (e.g. on the desk top).
 *
 * returns: non-zero if writing sample variogram to file failed.
 */
/*int calc_variogram(VARIOGRAM *v /* pointer to VARIOGRAM structure *///,       KS replaced
/*		const char *fname /* pointer to output file name, or NULL if
/*		no output has to be written to file */ //)
int calc_variogram(VARIOGRAM *v, CEFile *f)
{
	DATA **d = NULL;
//	FILE *f = NULL;

	assert(v);

	d = get_data();
	if (v->fname && (d[v->id1]->dummy || d[v->id2]->dummy)) {
		if (NULL == (v->ev = load_ev(v->ev, v->fname)))
			ErrMsg(ER_READ, "could not read sample variogram");
		v->ev->cloud = 0;
		v->ev->recalc = 0;
		return 0;
	}
	if (v->ev->evt == CROSSVARIOGRAM && v->ev->is_asym == -1) {
		/* v's first time */
		if (coordinates_are_equal(d[v->id1], d[v->id2]))
			v->ev->pseudo = 0;
		else
			v->ev->pseudo = 1;
		if (gl_sym_ev == 0)
			v->ev->is_asym = v->ev->pseudo;
			/* pseudo: always, else: only if set */
		else
			v->ev->is_asym = 0;
	}
	if (gl_zero_est == ZERO_DEFAULT) { /* choose a suitable default */
		if (is_covariogram(v))
			v->ev->zero = ZERO_SPECIAL;
		else { /* v is variogram */
			if (v->ev->pseudo)
				v->ev->zero = ZERO_SPECIAL;
			else
				v->ev->zero = ZERO_INCLUDE;
		}
	} else
		v->ev->zero = gl_zero_est;

	assert(v->ev->zero != ZERO_DEFAULT);

	fill_cutoff_width(d[v->id1], v);

	v->ev->cloud = (gl_iwidth <= 0.0); /*KS changed*//*v->ev->gl_width*/
	if (v->ev->cloud &&
			 (d[v->id1]->n_list >= MAX_NH ||  d[v->id2]->n_list >= MAX_NH))
		pr_warning("observation numbers in cloud will be wrong");
	set_direction_values(gl_alpha, gl_beta, gl_tol_hor, gl_tol_ver);

	v->ev->is_directional = is_directional(v);
	if (v->ev->recalc) {
		switch (v->ev->evt) {
			case SEMIVARIOGRAM:
				/*semivariogram(d[v->id1], v->ev);*/
                c_Semivariogram(d[v->id1],d[v->id2], v->ev,f);
				break;
			case CROSSVARIOGRAM:
				/*cross_variogram(d[v->id1], d[v->id2], v->ev);*/
                c_Cross_variogram(d[v->id1],d[v->id2], v->ev,f);
				break;
			case COVARIOGRAM:
				v->ev->is_asym = gl_sym_ev;
				/*covariogram(d[v->id1], v->ev);*/
                c_Covariogram(d[v->id1],d[v->id2], v->ev,f);
				break;
			case CROSSCOVARIOGRAM:
				/*cross_covariogram(d[v->id1], d[v->id2], v->ev);*/
                c_Cross_covariogram(d[v->id1], d[v->id2], v->ev,f);
				break;
			case NOTSPECIFIED:
			default:
				assert(0); /* aborts */
				break;
		}
	}
	/*KS*/d[v->id1]->colnx = 1; d[v->id1]->colny = 2; d[v->id1]->colnvalue = 3;    /*KS for Idrisi32 only,need condition*/
          d[v->id2]->colnx = 1; d[v->id2]->colny = 2; d[v->id2]->colnvalue = 3;

    if (v->ev->map)
		ev2map(v);
	/*else if (fname /*!= NULL*///) {
	/*	f = efopen(fname, "w");
		fprint_header_vgm(f, d[v->id1], d[v->id2], v->ev);
		fprint_sample_vgm(f, v->ev);
	} */
    /*else if ( (v->ev->evt !=  SEMIVARIOGRAM) &&
		(v->ev->evt  !=  CROSSVARIOGRAM) &&
		(f != NULL))
	{
		fprint_header_vgm(f, d[v->id1], d[v->id2], v->ev);
		fprint_sample_vgm(f, v->ev, d[v->id1]->n_list,d[v->id2]->n_list); /*KS added d->n_list*/
	/*} */

/*	if (f && f != stdout)
		return efclose(f);
	return 0;*//*KS removed*/
}

/*static SAMPLE_VGM *semivariogram(DATA *d, SAMPLE_VGM *ev) {
/*
 *  calculate sample variogram of 0.5 E[(Z(x)-Z(x+h))2]
 */
/*	ev->evt = SEMIVARIOGRAM;
	ev = alloc_exp_variogram(d, NULL, ev);
	if (d->grid /*!= NULL*///)
/*		ev = semivariogram_grid(d, ev);
	else
		ev = semivariogram_list(d, ev);
	divide(ev);
	ev->recalc = 0;
	return ev;
} /* semivariogram() */

/*static SAMPLE_VGM *semivariogram_list(DATA *d, SAMPLE_VGM *ev) {
	unsigned long uli, ulj;
	int i, j, index = 0;
	unsigned int total_steps;
	double gamma, ddist;

	total_steps = d->n_list * (d->n_list - 1) / 2;
	print_progress(0, total_steps);

	if (DEBUG_DUMP)
		printlog("Calculating semivariogram from %d points...\n", d->n_list);
	for (i = 0; i < d->n_list; i++) {
		print_progress(i * (i - 1) / 2, total_steps);
		for (j = 0; j < (ev->map /*!= NULL*/ //? d->n_list : i); j++) {
/*			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0 && i != j) {
				if (! ev->cloud) {
					index = get_index(ddist, ev);
					if (gl_cressie)  /* sqrt abs diff */
/*						ev->gamma[index] +=
							sqrt(fabs(d->list[i]->attr - d->list[j]->attr));
					else {
						ev->gamma[index] +=
							SQR(d->list[i]->attr - d->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (d->colnvariance)
							ev->gamma[index] -= d->list[i]->variance +
									 d->list[j]->variance;
#endif
					}
					ev->dist[index] += ddist;
					ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
						ev->nh[index], d->list[i], d->list[j]);
					ev->nh[index]++;
				} else { /* cloud: */
/*					if (! (ev->zero == ZERO_AVOID && ddist == 0.0)) {
						if (gl_cressie)
							gamma = sqrt(fabs(d->list[i]->attr - d->list[j]->attr));
						else
							gamma = SQR(d->list[i]->attr - d->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (d->colnvariance)
							gamma -= d->list[i]->variance + d->list[j]->variance;
#endif
						uli = i; ulj = j;
						push_to_cloud(ev, gamma / 2.0, ddist, TO_NH(uli,ulj));
					}
				}
			}/* if ddist >= 0 */
/*		}  /* for j */
/*	} /* for i */
/*	print_progress(total_steps, total_steps);
	if (DEBUG_DUMP)
		printlog("ready\n");
	return ev;
}

static SAMPLE_VGM *semivariogram_grid(DATA *d, SAMPLE_VGM *ev) {

	typedef struct {
		int row, col, ev_index;
		double dist;
	} grid_index;
	struct {
		int n;
		grid_index *gi;
	} grid_ev;
	int row, col, irow, icol, i, max, index;
	unsigned long ula, ulb;
	double gamma, ddist;
	DPOINT a, b, *dpa = NULL, *dpb = NULL;

	max = floor(ev->cutoff / d->grid->cellsize);
	grid_ev.gi = (grid_index *) emalloc(2 * (max + 1) * (max + 1)
		* sizeof(grid_index));
	grid_ev.n = 0;
	a.x = a.y = a.z = 0;
	for (row = 0; row <= max; row++) {
		for (col = (row == 0 ? 1 : -max); col <= max; col++) {
			b.x = row * d->grid->cellsize;
			b.y = col * d->grid->cellsize;
			ddist = valid_distance(&a, &b, ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist > 0.0) {
				i = grid_ev.n;
				grid_ev.gi[i].row = row;
				grid_ev.gi[i].col = col;
				grid_ev.gi[i].dist = ddist;
				if (! ev->cloud)
					grid_ev.gi[i].ev_index = get_index(ddist, ev);
				/* printf("row %d col %d index %d\n", row, col, index); */
/*				grid_ev.n++;
			}
		}
	}
	for (row = 0; row < d->grid->rows; row++) {
		for (col = 0; col < d->grid->cols; col++) {
			if (NULL != (dpa = d->grid->dpt[row][col]) /*!= NULL*///) {
/*				for (i = 0; i < grid_ev.n; i++) {
					irow = row + grid_ev.gi[i].row;
					icol = col + grid_ev.gi[i].col;
					if (irow >= 0 && icol >= 0 && irow < d->grid->rows
							&& icol < d->grid->cols
							&& (NULL != (dpb = d->grid->dpt[irow][icol]) /*!= NULL*///)) {
/*						ddist = grid_ev.gi[i].dist;
						if (! ev->cloud) {
							index = grid_ev.gi[i].ev_index;
							if (gl_cressie)  /* sqrt abs diff */
/*								ev->gamma[index] +=
									sqrt(fabs(dpa->attr - dpb->attr));
							else {
								ev->gamma[index] += SQR(dpa->attr - dpb->attr);
#ifdef ADJUST_VARIANCE
								if (d->colnvariance)
									ev->gamma[index] -= dpa->variance +
										 	dpb->variance;
#endif
							}
							ev->dist[index] += ddist;
							ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
								ev->nh[index], dpa, dpb);
							ev->nh[index]++;
						} else { /* cloud: */
/*							if (gl_cressie)
								gamma = sqrt(fabs(dpa->attr - dpb->attr));
							else
								gamma = SQR(dpa->attr - dpb->attr);
#ifdef ADJUST_VARIANCE
							if (d->colnvariance)
								gamma -= dpa->variance + dpb->variance;
#endif
							ula = GET_INDEX(dpa);
							ulb = GET_INDEX(dpb);
							push_to_cloud(ev, gamma / 2.0, ddist, TO_NH(ula,ulb));
						} /* else !cloud */
/*					} /* if we have two non-NULL points */
/*				} /* for all possibly relevant pairs */
/*			} /* if this grid cell is non-NULL */
/*		} /* for all cols */
/*	} /* for all rows */
/*	efree(grid_ev.gi);
	return ev;
} */

SAMPLE_VGM * UpFrontSemivariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2)
{
/*
 *  calculate sample variogram of 0.5 E[(Z(x)-Z(x+h))2]
 */
	int i, j,index = 0;
	double ddist;
    double hindex; /*KS added*/
    static int count = sizeof(double);
    double attrib;
    CEFile2      f_file;
    CEFile2*     fromf = &f_file;
    CEFile2      t_file;
    CEFile2*     tof = &t_file;
    char* pathtmp;

    	pathtmp = (char *) emalloc(strlen(directory)+9);
    	strcpy(pathtmp,directory); strcat(pathtmp,"from.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
		fromf->efopen1(pathtmp, "wb");
    	strcpy(pathtmp,directory); strcat(pathtmp,"to.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
    	tof->efopen3(pathtmp, "wb");

	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j < (ev->map != NULL ? d->n_list : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0 && i != j)
			{
				if (! ev->cloud)
				{
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
					{
						if (gl_cressie)
						{ /* sqrt abs diff */
	   						ev->gamma_last +=
								sqrt(fabs(d->list[i]->attr - d->list[j]->attr));

						}
						else
						{
							ev->gamma_last +=
								SQR(d->list[i]->attr - d->list[j]->attr);
#ifdef ADJUST_VARIANCE
							if (d->colnvariance)
	 						ev->gamma_last -= d->list[i]->variance +
										 d->list[j]->variance;
#endif
						}
                        attrib = (double )d->list[i]->attr; f2->fwrite(&attrib,count,1);  /*KS added*/
                        attrib = (double )d->list[j]->attr; f2->fwrite(&attrib,count,1);  /*KS added*/
                        hindex = 999; f2->fwrite(&hindex,count,1);  /*KS added*/
                        fromf->fwrite1(&d->list[i]->x,8,1); fromf->fwrite1(&d->list[i]->y,8,1);
                        tof->fwrite3(&d->list[j]->x,8,1); tof->fwrite3(&d->list[j]->y,8,1);
						ev->nh_last++;
					}
					else//(ddist == 0.0 && ev->zero != ZERO_INCLUDE)
					{

						index = get_index(ddist, ev->iwidth);
						if(  index  == ev->n_est -1 )
						{

							if (gl_cressie)
							{ /* sqrt abs diff */
								ev->gamma_last +=
									sqrt(fabs(d->list[i]->attr - d->list[j]->attr));


							}
							else
							{
							ev->gamma_last +=
									SQR(d->list[i]->attr - d->list[j]->attr);
#ifdef ADJUST_VARIANCE
							if (d->colnvariance)
								ev->gamma_last -= d->list[i]->variance +
											 d->list[j]->variance;

#endif
							}
							ev->dist_last += ddist;
							ev->nh_last++;
						}
						ev->distx[i] = (d->list[j]->x - d->list[i]->x);
                     	ev->disty[i] = (d->list[i]->y - d->list[j]->y);/*KS*/
                        attrib = (double )d->list[i]->attr; f2->fwrite(&attrib,count,1);  /*KS added*/
                        attrib = (double )d->list[j]->attr; f2->fwrite(&attrib,count,1);  /*KS added*/
                        hindex = (double )index;  f2->fwrite(&hindex,count,1);  /*KS added*/

						fromf->fwrite1(&d->list[i]->x,8,1); fromf->fwrite1(&d->list[i]->y,8,1);
                        tof->fwrite3(&d->list[j]->x,8,1); tof->fwrite3(&d->list[j]->y,8,1);
					}//end(ddist == 0.0 && ev->zero != ZERO_INCLUDE)

				}
				else
				{
					index++;
				}

			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	if (! ev->cloud)
	{
		f2->commit2(); fromf->commit2(); tof->commit2();
	}
    fromf->efclose2(); tof->efclose2();
    efree(pathtmp);
	return ev;
}

SAMPLE_VGM * PageSemivariogram(DATA *d, SAMPLE_VGM *ev, CEFile* f2)
{
/*
 *  calculate sample variogram of 0.5 E[(Z(x)-Z(x+h))2]
 */
	int i, j,index = 0;
	double ddist;
    //double hindex; /*KS added*/
    int k = -1;

    static int count = sizeof(float);

	int inbound  = ev->m_pos * ev->m_memsize;
	int outbound = (inbound + ev->m_memsize > ev->n_est) ?  ev->n_est: inbound + ev->m_memsize;

	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j < (ev->map != NULL ? d->n_list : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0 && i != j)
			{
				if (! ev->cloud)
				{
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
					{

					}
					else
					{
						index = get_index(ddist, ev->iwidth);
						if(  ( index  <  inbound )||(index >= outbound) )
						{
							continue;
						}
                        k = index - inbound;
						if (gl_cressie)
						{ /* sqrt abs diff */
							ev->gamma[k] +=
								sqrt(fabs(d->list[i]->attr - d->list[j]->attr));


                        }
						else
						{
						ev->gamma[k] +=
								SQR(d->list[i]->attr - d->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (d->colnvariance)
							ev->gamma[k] -= d->list[i]->variance +
										 d->list[j]->variance;

#endif
						}
						ev->dist[k] += ddist;
						ev->nh[k]++;
					}
				}
				else
				{ /* cloud: */

					if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
					{
						if( index >= outbound)
						{
							//f2->commit();
							return ev;
						}
						if(   index  <  inbound )
						{
                    			index++;
								continue;
						}
						if( k < 0)
						{
                    		k = index - inbound;
						}
						if (gl_cressie)
							ev->gamma[k] = sqrt(fabs(d->list[i]->attr -
									 d->list[j]->attr));
						else
							ev->gamma[k] = SQR(d->list[i]->attr -
									 d->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (d->colnvariance)
							ev->gamma[k] -= d->list[i]->variance +
									 d->list[j]->variance;
#endif
/*KS added*/            ev->distx[k] = (d->list[i]->x - d->list[j]->x);
                     	ev->disty[k] = (d->list[i]->y - d->list[j]->y); /*KS*/

            			ev->gamma[k] /= 2.0;
						ev->dist[k] = ddist;
						ev->nh[k] = TO_NH(i,j);
						index++;
                        k++;
					}
				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
    //f2->commit();
	if (!ev->cloud) PageDivide(ev);
	return ev;
}


SAMPLE_VGM * c_Semivariogram(DATA *d, DATA *b,SAMPLE_VGM *ev,CEFile* f)
{
	/*
 *  calculate sample variogram of 0.5 E[(Z(x)-Z(x+h))2]
 */
	CEFile localf;
    CEFile* f2 = &localf;
    char *path2,*cat; /*KS added*/
    int subtotal,count; /*KS added*/
    float xyz;

	ev->evt = SEMIVARIOGRAM;
	SetEVnest(d, NULL, ev);
	AllocVGM(d, NULL, ev);
	//ResetEVspace(ev);

	/*KS*/
	cat = (char *) emalloc(12);
    strcpy(cat,"idrtmp2.id$"); count = 2 + strlen(directory)+strlen(cat);
    path2 = (char *) emalloc(count);
    strcpy(path2,directory); strcat(path2,"idrtmp2.id$");
    if (file_exists(path2)) eremove(path2);
	f2->efopen2(path2, "wb");
    count = sizeof(float); /*KS*/
//    perc1 = 2; perc2 = 100;
//    percent_done(perc1,perc2,handle);
    if(!ev->cloud )
    {
    	UpFrontSemivariogram(d,ev,f2);
    }
    else
    {
    	ev->n_est = GetIndex(d,ev);
    }
	while(ev->m_pos * ev->m_memsize < ev->n_est )
	{
    	PageSemivariogram(d,ev,f2);
		if( 0 == ev->m_pos)
		{
       		d->colnx = 1; d->colny = 2; d->colnvalue = 3;
			fprint_header_vgm(f,d, b, ev);
		}
		if (!ev->cloud) Pagefprint_sample_vgm(f,ev, 0, 0);
        		else Pagefprint_sample_vgm(f2,ev, 0, 0);
        if (((ev->m_pos+1)*ev->m_memsize) < ev->n_est)
         	ResetEVspace(ev);/*KS set condition 1/18/999*/
/*KS*/  if (ev->m_memsize > ev->n_est) subtotal = (ev->n_est*0.85);
            else subtotal = ev->m_pos * ev->m_memsize;
        xyz = ((float)subtotal/(float)ev->n_est);
        if (xyz < 0.9000000)
			percent_done(subtotal, ev->n_est,handle); /*KS added 1/7/99*/
		ev->m_pos++;
	}
	f2->efclose2();
	ev->recalc = 0;  //I don't know how this is use
	return ev;
}



/* covariograms: */
/*static SAMPLE_VGM *covariogram(DATA *d, SAMPLE_VGM *ev) {
	int i, j, index = 0;
	unsigned long uli, ulj;
	double gamma, ddist;

	ev->evt = COVARIOGRAM;
	ev = alloc_exp_variogram(d, NULL, ev);
	for (i = 0; i < d->n_list; i++) {
		for (j = 0; j <= (ev->map /*!= NULL*/ //? d->n_list-1 : i); j++) {
/*			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0) {
				if (! ev->cloud) {
					index = get_index(ddist, ev);
					ev->gamma[index] += d->list[i]->attr * d->list[j]->attr;
#ifdef ADJUST_VARIANCE
					if (d->colnvariance && i == j)
						ev->gamma[index] -= d->list[i]->variance;
#endif
					ev->dist[index] += ddist;
					ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
						ev->nh[index], d->list[i], d->list[j]);
					ev->nh[index]++;
				} else {
					if (! (ev->zero == ZERO_AVOID && ddist == 0.0)) {
						gamma = d->list[i]->attr * d->list[j]->attr;
#ifdef ADJUST_VARIANCE
						if (d->colnvariance && i == j)
							gamma -= d->list[i]->variance;
#endif
						uli = i;
						ulj = j;
						push_to_cloud(ev, gamma, ddist, TO_NH(uli,ulj));
					}
				}
			}/* if ddist >= 0 */
/*		}  /* for j */
/*	} /* for i */
/*	divide(ev);
	ev->recalc = 0;
	return ev;
}*/ /* covariogram() */

static int GetMax_CovariogramIndex(DATA *d, SAMPLE_VGM *ev)
{
	int i, j, index = 0;
	double ddist;

	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j <= (ev->map != NULL ? d->n_list-1 : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
			{
				if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
				{
					index++;
				}

			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
    return index;
}

SAMPLE_VGM *UpFrontCovariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2)
{

	int i, j,index = 0;
	double ddist;
    double hindex; /*KS added*/
    static int count = sizeof(double);
    CEFile2      f_file;
    CEFile2*     fromf = &f_file;
    CEFile2      t_file;
    CEFile2*     tof = &t_file;
    char* pathtmp;

    	pathtmp = (char *) emalloc(strlen(directory)+9);
    	strcpy(pathtmp,directory); strcat(pathtmp,"from.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
		fromf->efopen1(pathtmp, "wb");
    	strcpy(pathtmp,directory); strcat(pathtmp,"to.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
    	tof->efopen3(pathtmp, "wb");

	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j <= (ev->map != NULL ? d->n_list-1 : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
			{
				if (! ev->cloud)
				{
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
					{
       				 	ev->gamma_last +=
							d->list[i]->attr * d->list[j]->attr;
#ifdef ADJUST_VARIANCE
						if (d->colnvariance && i == j)
							ev->gamma_last -= d->list[i]->variance;
#endif
                          ev->Xatt_n_est += d->list[i]->attr;  /*KS */
                          ev->Yatt_n_est += d->list[j]->attr;  /*KS */
                        if (CORRELOGRAM) {
                          ev->SQRXatt_n_est += SQR(d->list[i]->attr);  /*KS */
                          ev->SQRYatt_n_est += SQR(d->list[j]->attr);  /*KS */
                        }
                        ev->nh_last++;

                        hindex = (double)d->list[i]->attr; f2->fwrite(&hindex,count,1);
                        hindex = (double)d->list[j]->attr; f2->fwrite(&hindex,count,1);
                        hindex = 999; f2->fwrite(&hindex,count,1);

                        fromf->fwrite1(&d->list[i]->x,8,1); fromf->fwrite1(&d->list[i]->y,8,1);
                        tof->fwrite3(&d->list[j]->x,8,1); tof->fwrite3(&d->list[j]->y,8,1);
					}
					else
					{
						index = get_index(ddist, ev->iwidth);
						if (index == ev->n_est)
						{
							ev->Xatt_n_est += d->list[i]->attr;
                     		ev->Yatt_n_est += d->list[j]->attr;
                         if (CORRELOGRAM) {
                            ev->SQRXatt_n_est += SQR(d->list[i]->attr);  /*KS */
                            ev->SQRYatt_n_est += SQR(d->list[j]->attr);  /*KS */
                         }
						}
						else if (index == ev->n_est -1)
						{
							ev->nh_last++;
						}
                        hindex = (double)d->list[i]->attr; f2->fwrite(&hindex,count,1);
                        hindex = (double)d->list[j]->attr; f2->fwrite(&hindex,count,1);
						hindex = (double)(index);f2->fwrite(&hindex,count,1);

                        fromf->fwrite1(&d->list[i]->x,8,1); fromf->fwrite1(&d->list[i]->y,8,1);
                        tof->fwrite3(&d->list[j]->x,8,1); tof->fwrite3(&d->list[j]->y,8,1);
					}

				}
				else
				{

				}
			}/* if ddist >= 0 */
		}  /* for j */
    /*KS*/  if (((float)i/(float)d->n_list)<=0.85)
			percent_done(i, d->n_list,handle);/*KS added 1/7/99*/
	} /* for i */
    if (! ev->cloud)
	{
		f2->commit2();
        fromf->commit2();
        tof->commit2();
	}
    fromf->efclose2();
    tof->efclose2();
    efree(pathtmp);
	return ev;
} /* covariogram() */


SAMPLE_VGM *PageCovariogram(DATA *d, SAMPLE_VGM *ev,CEFile* f2)
{
	int i, j,index = 0;
	double ddist;
   // double hindex; /*KS added*/
    int k = -1;

    static int count = sizeof(double);
	int inbound  = ev->m_pos * ev->m_memsize;
	int outbound = (inbound + ev->m_memsize > ev->n_est) ?  ev->n_est: inbound + ev->m_memsize;
	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j <= (ev->map != NULL ? d->n_list-1 : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
			{
				if (! ev->cloud)
				{
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
					{

					}
					else
					{
						index = get_index(ddist, ev->iwidth);
						if(  ( index  <  inbound )||(index >= outbound) )
						{
							continue;
						}
						k = index - inbound;
						ev->gamma[k] += d->list[i]->attr * d->list[j]->attr;
#ifdef ADJUST_VARIANCE
						if (d->colnvariance && i == j)
						ev->gamma[k] -= d->list[i]->variance;
#endif
                          ev->Xatt[k] += d->list[i]->attr;  /*KS*/
                          ev->Yatt[k] += d->list[j]->attr;  /*KS*/
                        if (CORRELOGRAM) {
                          ev->SQRXatt[k] += SQR(d->list[i]->attr);  /*KS*/
                          ev->SQRYatt[k] += SQR(d->list[j]->attr);  /*KS*/
                        }

                        ev->dist[k] += ddist;
						ev->nh[k]++;
					}

				}
				else
				{
					if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
					{
						if( index >= outbound)
						{
							//f2->commit();
							return ev;
						}
						if(   index  <  inbound )
						{
                    			index++;
								continue;
						}
						if( k < 0)
						{
                    		k = index - inbound;
						}
						ev->gamma[k] = d->list[i]->attr * d->list[j]->attr;
#ifdef ADJUST_VARIANCE
						if (d->colnvariance && i == j)
							ev->gamma[k] -= d->list[i]->variance;
#endif
/*KS added*/            //if (ev->nh[k]!= 0)
						//	ev->gamma[k] /= (1.0 * ev->nh[index]);
						ev->distx[k] = (d->list[i]->x - d->list[j]->x);
                     	ev->disty[k] = (d->list[i]->y - d->list[j]->y); /*KS*/

						ev->dist[k] = ddist;
						ev->nh[k] = TO_NH(i,j);
						index++;
						k++;
					}
				}
			}/* if ddist >= 0 */
		}  /* for j */
    /*KS*/  if (((float)i/(float)d->n_list)<=0.85)
			percent_done(i, d->n_list,handle);/*KS added 1/7/99*/
	} /* for i */
    if (!ev->cloud) PageDivide(ev);
	return ev;
} /* covariogram() */

SAMPLE_VGM *c_Covariogram(DATA *d,DATA *b, SAMPLE_VGM *ev,CEFile* f)
{
	CEFile ff;
    CEFile* f2 = &ff; //pointer is use to make the less change as possible
    char *path;
    int subtotal,count; /*KS added*/
    float xyz; /*KS added*/

	ev->evt = COVARIOGRAM;
	SetEVnest(d, NULL, ev);
	AllocVGM(d, NULL, ev);
	//ResetEVspace(ev);

/*KS*/
	count = 12 + strlen(directory);
    path = (char *) emalloc(count);
    strcpy(path,directory); strcat(path,"idrtmp2.id$");
    if (file_exists(path)) eremove(path);
    f2->efopen2(path, "wb");
    count = sizeof(double); /*KS*/
//    perc1 = 2; perc2 = 100;
//    percent_done(perc1,perc2,handle);
	if(!ev->cloud )
    {
    	UpFrontCovariogram(d,ev,f2);
    }
    else
    {
    	ev->n_est = GetMax_CovariogramIndex(d,ev);
    }
	while(ev->m_pos * ev->m_memsize < ev->n_est )
	{
    	PageCovariogram(d,ev,f2);
		if( 0 == ev->m_pos)
		{
        	/*if (d->type == MT_IDRISI)
            { */
          		d->colnx = 1; d->colny = 2; d->colnvalue = 3;
          	//}  /*KS added*/
			/*if(!ev->cloud )*/ fprint_header_vgm(f,d, b, ev);
		}
		if (!ev->cloud) Pagefprint_sample_vgm(f,ev, 0, 0); else Pagefprint_sample_vgm(f2,ev, 0, 0);
		if (ev->iwidth == 0)
         	ResetEVspace(ev);/*KS set condition 1/18/999*/
/*KS*/  if (ev->m_memsize > ev->n_est) subtotal = (ev->n_est*0.85);
            else subtotal = ev->m_pos * ev->m_memsize;
        xyz = ((float)subtotal/(float)ev->n_est);
        if (xyz < 0.9000000)
			percent_done(subtotal, ev->n_est,handle); /*KS added 1/7/99*/
		ev->m_pos++;
	}
	f2->efclose2();
    efree(path);
	ev->recalc = 0;  //I don't know how this is use
	return ev;
} /* covariogram() */


/*static SAMPLE_VGM *cross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev) {
	int i, j, index = 0;
	unsigned long uli, ulj;
	double gamma, ddist;

	ev->evt = CROSSVARIOGRAM;
	ev = alloc_exp_variogram(a, b, ev);
	for (i = 0; i < a->n_list; i++) {
		for (j = 0; j < b->n_list; j++) {
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev || !ev->pseudo, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0) {
				if (!ev->pseudo && i != j) {
					if (! ev->cloud) {
						index = get_index(ddist, ev);
						ev->gamma[index] +=
							(a->list[i]->attr - a->list[j]->attr) *
							(b->list[i]->attr - b->list[j]->attr);
						ev->dist[index] += ddist;
						ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
							ev->nh[index], a->list[i], a->list[j]);
						ev->nh[index]++;
					} else if (!(ddist == 0.0 && ev->zero == ZERO_AVOID)) {
						gamma = (a->list[i]->attr - a->list[j]->attr) *
							(b->list[i]->attr - b->list[j]->attr);
						uli = i;
						ulj = j;
						push_to_cloud(ev, gamma / 2.0, ddist, TO_NH(uli,ulj));
					}
				} else if (ev->pseudo) {
					if (! ev->cloud) {
						index = get_index(ddist, ev);
						ev->gamma[index] +=
							SQR(a->list[i]->attr - b->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (a->colnvariance || b->colnvariance)
							ev->gamma[index] -= a->list[i]->variance +
								 b->list[j]->variance;
#endif
						ev->dist[index] += ddist;
						ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
							ev->nh[index], a->list[i], b->list[j]);
						ev->nh[index]++;
					} else if (! (ev->zero == ZERO_AVOID && ddist == 0.0)) {
						gamma = SQR(a->list[i]->attr - b->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (a->colnvariance || b->colnvariance)
							gamma -= a->list[i]->variance + b->list[j]->variance;
#endif
						uli = i;
						ulj = j;
						push_to_cloud(ev, gamma / 2.0, ddist, TO_NH(uli,ulj));
					}
				}
			}/* if ddist >= 0 */
/*		}  /* for j */
/*	} /* for i */
/*	divide(ev);
	ev->recalc = 0;
	return ev;
} /* cross_variogram */

static int GeMaxCross_variogramIndex(DATA *a, DATA *b, SAMPLE_VGM *ev)
{
	int i, j, index = 0;
	double ddist;

	for (i = 0; i < a->n_list; i++)
    {
		for (j = 0; j < b->n_list; j++)
        {
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev || !ev->pseudo, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {
				if (!ev->pseudo && i != j)
                {
                    if (!(ddist == 0.0 && ev->zero == ZERO_AVOID))
                    {
						index++;
					}
				}
				else if (ev->pseudo)
				{
					 if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
					 {
						index++;
					 }
				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	return index;
}

SAMPLE_VGM *PageCross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f2)
{

	int i, j,index = 0;
	double ddist;
    int k = -1;

    static int count = sizeof(double);

	int inbound  = ev->m_pos * ev->m_memsize;
	int outbound = (inbound + ev->m_memsize > ev->n_est) ?  ev->n_est: inbound + ev->m_memsize;
	for (i = 0; i < a->n_list; i++)
    {
		for (j = 0; j < b->n_list; j++)
        {
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev || !ev->pseudo, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {
				if (!ev->pseudo && i != j)
                {
					if (! ev->cloud)
                    {
						if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                        {

						}
                        else
                        {
							index = get_index(ddist, ev->iwidth);
							if(  ( index  <  inbound )||(index >= outbound) )
							{
								continue;
							}
							k = index - inbound;
		   					ev->gamma[k] +=
								(a->list[i]->attr - a->list[j]->attr) *
								(b->list[i]->attr - b->list[j]->attr);

                            ev->dist[k] += ddist;
							ev->nh[k]++;
						}
					}
                    else if (!(ddist == 0.0 && ev->zero == ZERO_AVOID))
                    {
						if( index >= outbound)
						{
							return ev;
						}
						if(   index  <  inbound )
						{
                    			index++;
								continue;
						}
						if( k < 0)
						{
                    		k = index - inbound;
						}
						ev->gamma[k] =
							0.5 * (a->list[i]->attr - a->list[j]->attr) *
							(b->list[i]->attr - b->list[j]->attr);

/*KS added*/            ev->distx[k] = (a->list[i]->x - b->list[j]->x);
                     	ev->disty[k] = (a->list[i]->y - b->list[j]->y); /*KS*/

						ev->dist[k] = ddist;
						ev->nh[k] = TO_NH(i,j);
						index++;
						 k++;
					}
				}
                 else if (ev->pseudo)
                 {
					if (! ev->cloud)
                    {
						if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                        {

						}
                        else
                        {
							index = get_index(ddist,ev->iwidth);
							if(  ( index  <  inbound )||(index >= outbound) )
							{
								continue;
							}
							k = index - inbound;
							ev->gamma[k] +=
								SQR(a->list[i]->attr - b->list[j]->attr);
#ifdef ADJUST_VARIANCE
							if (a->colnvariance || b->colnvariance)
							ev->gamma[k] -= a->list[i]->variance +
									 b->list[j]->variance;
#endif
                            ev->dist[k] += ddist;
							ev->nh[k]++;
						}
					}
                     else if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
                     {
						if( index >= outbound)
						{
							//f2->commit();
							return ev;
						}
						if(   index  <  inbound )
						{
                    			index++;
								continue;
						}
						if( k < 0)
						{
                    		k = index - inbound;
						}
						ev->gamma[k] =
							SQR(a->list[i]->attr - b->list[j]->attr);
#ifdef ADJUST_VARIANCE
						if (a->colnvariance || b->colnvariance)
							ev->gamma[k] -= a->list[i]->variance +
								 b->list[j]->variance;
#endif
/*KS added*/            ev->distx[k] = (a->list[i]->x - b->list[j]->x);
                     	ev->disty[k] = (a->list[i]->y - b->list[j]->y); /*KS*/

						ev->gamma[k] /= 2.0;
						ev->dist[k] = ddist;
						ev->nh[k] = TO_NH(i,j);
						index++;
						 k++;
					}
				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	//divide(ev);
	if (!ev->cloud) PageDivide(ev);
	return ev;
} /* cross_variogram */


SAMPLE_VGM *c_Cross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f)
{

	CEFile localf;
    CEFile* f2 = &localf;
    char *path; /*KS added*/
    int subtotal,count; /*KS added*/
    float xyz; /*KS added*/

	ev->evt = CROSSVARIOGRAM;
	//ev = alloc_exp_variogram(a, b, ev);
    SetEVnest(a, b, ev);
	AllocVGM(a,b, ev);
	//ResetEVspace(ev);

	/*KS*/
     count = 12 + strlen(directory);
    path = (char *) emalloc(count);
    strcpy(path,directory); strcat(path,"idrtmp2.id$");
    if (file_exists(path)) eremove(path);
	f2->efopen2(path, "wb");
    count = sizeof(double); /*KS added*/
//    perc1 = 2; perc2 = 100;
//    percent_done(perc1,perc2,handle);
    if(!ev->cloud )
    {
    	UpFrontCross_variogram(a,b,ev,f2);
    }
    else
    {
    	ev->n_est = GeMaxCross_variogramIndex(a,b,ev);
    }
	while(ev->m_pos * ev->m_memsize < ev->n_est )
	{
    	PageCross_variogram(a,b,ev,f2);
		if( 0 == ev->m_pos)
		{
        	/*if (a->type==DATA_IMG_BIN)
            { */
          		a->colnx = 1; a->colny = 2; a->colnvalue = 3;
          	//}  /*KS added*/
			/*if(!ev->cloud )*/ fprint_header_vgm(f,a, b, ev);
		}
		if (!ev->cloud) Pagefprint_sample_vgm(f,ev, 0, 0); else Pagefprint_sample_vgm(f2,ev, 0, 0);
		if (ev->iwidth == 0)
         	ResetEVspace(ev);/*KS set condition 1/18/999*/
/*KS*/  if (ev->m_memsize > ev->n_est) subtotal = (ev->n_est*0.85);
            else subtotal = ev->m_pos * ev->m_memsize;
        xyz = ((float)subtotal/(float)ev->n_est);
        if (xyz < 0.9000000)
			percent_done(subtotal, ev->n_est,handle); /*KS added 1/7/99*/
		ev->m_pos++;
	}
	f2->efclose2();
    efree(path);
	ev->recalc = 0;  //I don't know how this is use
	return ev;
} /* cross_variogram */


SAMPLE_VGM * UpFrontCross_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev,CEFile* f2)
{
	int i, j, index = 0;
	double ddist;
    double hindex;
    static int count = sizeof(double);
    CEFile2      f_file;
    CEFile2*     fromf = &f_file;
    CEFile2      t_file;
    CEFile2*     tof = &t_file;
    char* pathtmp;

    	pathtmp = (char *) emalloc(strlen(directory)+9);
    	strcpy(pathtmp,directory); strcat(pathtmp,"from.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
		fromf->efopen1(pathtmp, "wb");
    	strcpy(pathtmp,directory); strcat(pathtmp,"to.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
    	tof->efopen3(pathtmp, "wb");

	for (i = 0; i < a->n_list; i++)
    {
		for (j = 0; j < b->n_list; j++)
        {
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev || !ev->pseudo, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {
				if (!ev->pseudo && i != j)
                {
					if (! ev->cloud)
                    {
						if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                        {
							ev->gamma_last +=
								(a->list[i]->attr - a->list[j]->attr) *
								(b->list[i]->attr - b->list[j]->attr);
                            ev->nh_last++;

							hindex = (double) (a->list[i]->attr - a->list[j]->attr); f2->fwrite(&hindex,count,1);
                            hindex = (double) (b->list[i]->attr - b->list[j]->attr); f2->fwrite(&hindex,count,1);
                        	hindex = 999; f2->fwrite(&hindex,count,1);  /*KS added*/

                            fromf->fwrite1(&a->list[i]->x,8,1); fromf->fwrite1(&a->list[i]->y,8,1);
                        	tof->fwrite3(&b->list[j]->x,8,1); tof->fwrite3(&b->list[j]->y,8,1);
						}
						else
						{
							index = get_index(ddist, ev->iwidth);
							hindex = (double) (a->list[i]->attr - a->list[j]->attr); f2->fwrite(&hindex,count,1);
                            hindex = (double) (b->list[i]->attr - b->list[j]->attr); f2->fwrite(&hindex,count,1);
                            hindex = (double)(index); f2->fwrite(&hindex,count,1);  /*KS added*/

                            fromf->fwrite1(&a->list[i]->x,8,1); fromf->fwrite1(&a->list[i]->y,8,1);
                        	tof->fwrite3(&b->list[j]->x,8,1); tof->fwrite3(&b->list[j]->y,8,1);
						}
					}

				}
                 else if (ev->pseudo)
                 {
					if (! ev->cloud)
                    {
						if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                        {


						}
						else
						{
							index = get_index(ddist,ev->iwidth);
                            hindex = (double)a->list[i]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                            hindex = (double)b->list[j]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                        	hindex = (double)(index); f2->fwrite(&hindex,count,1);  /*KS added*/

                            fromf->fwrite1(&a->list[i]->x,8,1); fromf->fwrite1(&a->list[i]->y,8,1);
                            tof->fwrite3(&b->list[j]->x,8,1); tof->fwrite3(&b->list[j]->y,8,1);
						}
					}

				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	if (! ev->cloud)
	{
		f2->commit2();
        fromf->commit2();
        tof->commit2();
	}
    fromf->efclose2();
    tof->efclose2();
    efree(pathtmp);
	return ev;
}


/*static SAMPLE_VGM *cross_covariogram(DATA *a, DATA *b, SAMPLE_VGM *ev) {
	int i, j, index = 0;
	unsigned long uli, ulj;
	double gamma, ddist;

	ev->evt = CROSSCOVARIOGRAM;
	ev = alloc_exp_variogram(a, b, ev);
	for (i = 0; i < a->n_list; i++) {      /* i -> a */
/*		for (j = 0; j < b->n_list; j++) {  /* j -> b */
/*			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0) {
				if (! ev->cloud) {
					index = get_index(ddist, ev);
					ev->gamma[index] += a->list[i]->attr * b->list[j]->attr;
					ev->dist[index] += ddist;
					ev->pairs[index] = (DPOINT **) register_pairs(ev->pairs[index],
						ev->nh[index], a->list[i], b->list[j]);
					ev->nh[index]++;
				} else if (! (ev->zero == ZERO_AVOID && ddist == 0.0)) {
					gamma = a->list[i]->attr * b->list[j]->attr;
					uli = i;
					ulj = j;
					push_to_cloud(ev, gamma, ddist, TO_NH(uli,ulj));
				}
			}/* if ddist >= 0 */
/*		}  /* for j */
/*	} /* for i */
/*	divide(ev);
	ev->recalc = 0;
	return ev;
} /* cross_covariogram() */

static int GetMaxCross_covariogramIndex(DATA *a, DATA *b,SAMPLE_VGM *ev)
{
	int i, j,index = 0;
	double ddist;

	for (i = 0; i < a->n_list; i++)
    {      /* i -> a */
		for (j = 0; j < b->n_list; j++)
        {  /* j -> b */
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {	
                if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
                {
					index++;
				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	return index;
} /* cross_covariogram() */


SAMPLE_VGM * UpFrontCross_covariogram(DATA *a, DATA *b,SAMPLE_VGM *ev,CEFile* f2)
{
	int i, j,index = 0;
	double ddist;
    double hindex; /*KS added*/
    static int count = sizeof(double);
    CEFile2      f_file;
    CEFile2*     fromf = &f_file;
    CEFile2      t_file;
    CEFile2*     tof = &t_file;
    char* pathtmp;

    	pathtmp = (char *) emalloc(strlen(directory)+9);
    	strcpy(pathtmp,directory); strcat(pathtmp,"from.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
		fromf->efopen1(pathtmp, "wb");
    	strcpy(pathtmp,directory); strcat(pathtmp,"to.id$");
        if (file_exists(pathtmp)) eremove(pathtmp);
    	tof->efopen3(pathtmp, "wb");


	for (i = 0; i < a->n_list; i++)
    {      /* i -> a */
		for (j = 0; j < b->n_list; j++)
        {  /* j -> b */
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {
				if (! ev->cloud)
                {
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                    {
						ev->gamma_last += a->list[i]->attr
							* b->list[j]->attr;

          /*KS added*/
                        ev->Xatt_n_est += a->list[i]->attr;
                     	ev->Yatt_n_est += b->list[j]->attr;
                        if (CORRELOGRAM) {
                           ev->SQRXatt_n_est += SQR(a->list[i]->attr);
                     	   ev->SQRYatt_n_est += SQR(b->list[j]->attr);
                        }

                        hindex = (double)a->list[i]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                        hindex = (double)b->list[j]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                        hindex = 999;f2->fwrite(&hindex,count,1);  /*KS added*/

                        fromf->fwrite1(&a->list[i]->x,8,1); fromf->fwrite1(&a->list[i]->y,8,1);
                        tof->fwrite3(&b->list[j]->x,8,1); tof->fwrite3(&b->list[j]->y,8,1);
                        ev->nh_last++;
					}
                    else
                    {
						index = get_index(ddist, ev->iwidth);
						if(index == ev->n_est)
						{
          /*KS added*/
                           ev->Xatt_n_est += a->list[i]->attr;
                     	   ev->Yatt_n_est += b->list[j]->attr;
                         if (CORRELOGRAM) {
                           ev->SQRXatt_n_est += SQR(a->list[i]->attr);
                     	   ev->SQRYatt_n_est += SQR(b->list[j]->attr);
                         }
						}
						else if (index == ev->n_est -1)
						{
							ev->nh_last++;
						}

                        hindex = (double)a->list[i]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                        hindex = (double)b->list[j]->attr; f2->fwrite(&hindex,count,1);  /*KS added*/
                        hindex = (double)(index); f2->fwrite(&hindex,count,1);  /*KS added*/

                        fromf->fwrite1(&a->list[i]->x,8,1); fromf->fwrite1(&a->list[i]->y,8,1);
                        tof->fwrite3(&b->list[j]->x,8,1); tof->fwrite3(&b->list[j]->y,8,1);

					}
				}
                else if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
                {

				}
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
	//divide(ev);
    if (! ev->cloud)
	{
		f2->commit2(); fromf->commit2(); tof->commit2();
	}
    fromf->efclose2(); tof->efclose2();
    efree(pathtmp);

	return ev;
} /* cross_covariogram() */


SAMPLE_VGM * PageCross_covariogram(DATA *a, DATA *b,SAMPLE_VGM *ev,CEFile* f2)
{
	int i, j,index = 0;
	double ddist;
   // double hindex; /*KS added*/
    int k = -1;

    static int count = sizeof(double);

	int inbound  = ev->m_pos * ev->m_memsize;
	int outbound = (inbound + ev->m_memsize > ev->n_est) ?  ev->n_est: inbound + ev->m_memsize;

	for (i = 0; i < a->n_list; i++)
    {      /* i -> a */
		for (j = 0; j < b->n_list; j++)
        {  /* j -> b */
			ddist = valid_distance(a->list[i], b->list[j], ev->cutoff,
				gl_sym_ev, a, b, (GRIDMAP *) ev->map);
			if (ddist >= 0.0)
            {
				if (! ev->cloud)
                {
					if (ddist == 0.0 && ev->zero != ZERO_INCLUDE)
                    {

					}
                    else
                    {
						index = get_index(ddist, ev->iwidth);
						if(  ( index  <  inbound )||(index >= outbound) )
						{
							continue;
						}
						k = index - inbound;
       					ev->gamma[k] += a->list[i]->attr * b->list[j]->attr;

                           ev->Xatt[k] += a->list[i]->attr;  /*KS*/
                           ev->Yatt[k] += b->list[j]->attr;  /*KS*/
                        if (CORRELOGRAM) {
                           ev->SQRXatt[k] += SQR(a->list[i]->attr);  /*KS*/
                           ev->SQRYatt[k] += SQR(b->list[j]->attr);  /*KS*/
                        }

                        ev->dist[k] += ddist;
						ev->nh[k]++;
					}
				}
                else if (! (ev->zero == ZERO_AVOID && ddist == 0.0))
                {
					if( index >= outbound)
					{
						//f2->commit();
						return ev;
					}
					if(   index  <  inbound )
					{
                    		index++;
							continue;
					}
					if( k < 0)
					{
                    	k = index - inbound;
					}
					ev->gamma[k] = a->list[i]->attr * b->list[j]->attr;

/*KS added*/            if (ev->nh[k]!= 0)//where is this previously initialized?
						ev->gamma[k] /= (1.0 * ev->nh[k]);
			        ev->distx[k] = (a->list[i]->x - b->list[j]->x);
                    ev->disty[k] = (a->list[i]->y - b->list[j]->y); /*KS*/

					ev->dist[k] = ddist;
					ev->nh[k] = TO_NH(i,j);
					index++;
					k++;
				}
			}/* if ddist >= 0 */
		}  /* for j */
  /*KS*/  if (((float)i/(float)a->n_list)<=0.85)
			percent_done(i, a->n_list,handle);/*KS added 1/7/99*/
	} /* for i */
	//divide(ev);
	if (!ev->cloud) PageDivide(ev);
	return ev;
} /* cross_covariogram() */


SAMPLE_VGM * c_Cross_covariogram(DATA *a, DATA *b,SAMPLE_VGM *ev,CEFile* f)
{

    //FILE *f2;
    CEFile ff;
    CEFile* f2 = &ff; //pointer is use to make the less change as possible
    char *path;
    int subtotal,count; /*KS added*/

	ev->evt = CROSSCOVARIOGRAM;
	//ev = alloc_exp_variogram(a, b, ev);
    SetEVnest(a, b, ev);
	AllocVGM(a,b, ev);
	//ResetEVspace(ev);

/*KS*/  count = 12 + strlen(directory);
    path = (char *) emalloc(count);
    strcpy(path,directory); strcat(path,"idrtmp2.id$");
    if (file_exists(path)) eremove(path);
	f2->efopen2(path, "wb");
//    perc1 = 2; perc2 = 100;
//    percent_done(perc1,perc2,handle);
	if(!ev->cloud )
    {
    	UpFrontCross_covariogram(a,b,ev,f2);
    }
    else
    {
    	ev->n_est = GetMaxCross_covariogramIndex(a,b,ev);
    }
	while(ev->m_pos * ev->m_memsize < ev->n_est )
	{
    	PageCross_covariogram(a,b,ev,f2);
		if( 0 == ev->m_pos)
		{
        	/*if (a->type==DATA_IMG_BIN)
            { */
          		a->colnx = 1; a->colny = 2; a->colnvalue = 3;
          	//}  /*KS added*/
			fprint_header_vgm(f,a, b, ev);
		}
		if (!ev->cloud) Pagefprint_sample_vgm(f,ev, 0, 0); else Pagefprint_sample_vgm(f2,ev, 0, 0);
   		if (ev->iwidth == 0)
         	ResetEVspace(ev);/*KS set condition 1/18/999*/
/*KS*/ if (ev->m_memsize > ev->n_est) subtotal = (ev->n_est*0.85);
             else subtotal = ev->m_pos * ev->m_memsize;
            if (((float)subtotal/(float)ev->n_est)<=0.85)
			percent_done(subtotal, ev->n_est,handle);/*KS added 1/7/99*/
		ev->m_pos++;
	}
	f2->efclose2();
    efree(path);
	ev->recalc = 0;  //I don't know how this is use
	return ev;
} /* cross_covariogram() */

static double valid_distance(DPOINT *a, DPOINT *b, double max,
		int symmetric, DATA *d1, DATA *d2, GRIDMAP *map) {
	double ddist, dX, dX2, inprod;
	DPOINT p;
	int mode = 0, i;
	unsigned int row, col;

	assert(a /*!= NULL*/);
	assert(b /*!= NULL*/);
	assert(d1 /*!= NULL*/);
	assert(d2 /*!= NULL*/);

	mode = d1->mode & d2->mode;
/*
 * even if modes don't correspond, valid_direction() will 
 * calculate valid distances 
 */
	p.x = a->x - b->x;
	p.y = a->y - b->y;
	p.z = a->z - b->z;
	if (map) {
		/* transform here p to allow directional 2d cuts in a 3d world */
		if (map_xy2rowcol(map, p.x, p.y, &row, &col))
			return -1.0;
		else
			ddist = (1.0 * row) * map->cols + col + 0.5;
	} else {
		if (p.x > max || p.y > max || p.z > max) 
			return -1.0;
		/* Changed K.M. Fri Feb 27 15:56:57 1998 */
		/* if ddist < 0.0 then we don't need to check for dX! */
		if ((ddist = valid_direction(&p, symmetric, d1)) > max || ddist < 0.0) 
			return -1.0;
	}
	dX = MIN(d1->dX, d2->dX);
	if (dX < DBL_MAX) {
		dX2 = dX * dX;
		/* allow only points for which 2-norm ||x_i-x_j|| < dX */
		if (d1->n_X != d2->n_X)
			ErrMsg(ER_IMPOSVAL, "valid_distance(): d1->n_X != d2->n_X");
		for (i = 0, inprod = 0.0; i < d1->n_X; i++)
			inprod += SQR(a->X[i] - b->X[i]);
		if (inprod > dX2)
			ddist = -1.0;
	}
	/*
	if (d1->coln_id > 0 && d2->coln_id > 0 && strcmp())
		return -1.0;
	*/
	return ddist;
}

int is_directional(VARIOGRAM *v) {
	switch(v->ev->evt) {
		case CROSSCOVARIOGRAM:
			if (gl_sym_ev == 0) /* asymmetric cross(co)variances: */
				return (gl_tol_hor < 180.0 || gl_tol_ver < 180.0);
			else
				return (gl_tol_hor < 90.0 || gl_tol_ver < 90.0);
		case CROSSVARIOGRAM:
			if (v->ev->is_asym && gl_sym_ev == 0) /* asymm. cross(co)variances: */
				return (gl_tol_hor < 180.0 || gl_tol_ver < 180.0);
			else
				return (gl_tol_hor < 90.0 || gl_tol_ver < 90.0);
		default: /* symmetric (co)variances */
			return (gl_tol_hor < 90.0 || gl_tol_ver < 90.0);
	}
	assert(0);
	return 0; /* never reached */
}

/*
 * this function should be changed--the mask map stack is misused as
 * to define the topology of variogram maps.
 *
 * use min/max coordinates for block diagonal as maximum cutoff
 * Returns: about 1/3 the max. dist between any two points in data. 
 */
void fill_cutoff_width(DATA *data /* pointer to DATA structure to derive
		the values from */,
		VARIOGRAM *v /* pointer to VARIOGRAM structure */)
{
	double d = 0.0;
	int i;
	GRIDMAP *m;
	SAMPLE_VGM *ev;

	assert(data);
	assert(v);

	ev = v->ev;
	if (get_n_masks() > 0) {
		m = new_map();
		m->is_write = 0;
		m->filename = get_mask_name(0);
		if (NULL == (m = map_read(m)))
			ErrMsg(ER_READ, "cannot open map");
		ev->iwidth = 1.0;
		ev->cutoff = m->rows * m->cols;
		ev->map = m;
		if (!v->fname /*== NULL*/)
			pr_warning("variogram map name not set");
	} else if (gl_bounds /*!= NULL*/) {
		i = 0;
		while (gl_bounds[i] >= 0.0) /* count length */
			i++;
		ev->cutoff = gl_bounds[i-1];
		ev->iwidth = ev->cutoff / i;
	} else {
		if (is_mv_double(&(ev->cutoff))) {
			if (gl_cutoff < 0.0) {
				d = data_block_diagonal(data);
				if (d == 0.0)
					ev->cutoff = 1.0; /* ha ha ha */
				else
					ev->cutoff = d * gl_fraction;
			} else
				ev->cutoff = gl_cutoff;
		}
		if (is_mv_double(&(ev->iwidth))) {
			if (gl_iwidth < 0.0)
				ev->iwidth = ev->cutoff / gl_n_intervals;
			else
				ev->iwidth = gl_iwidth;
		}
	}
}

/*void fprint_header_vgm(FILE *f, const DATA *a, const DATA *b,
		const SAMPLE_VGM *ev) {
	time_t tp;
	extern char *command_line;
	char *cp = NULL;
	/* char *pwd; */

/*	fprintf(f, "#gstat %s %s [%s]", GSTAT_OS, VERSION, command_line);
	/* if (pwd = getenv("PWD")) fprintf(f, "(in %s)", pwd); */
/*	fprintf(f, "\n");
	fprintf(f, "#sample %s%s\n",
		(ev->evt == CROSSVARIOGRAM && ev->pseudo ? "pseudo " : ""),
		vgm_type_str[ev->evt]);
	tp = time(&tp);
	fprintf(f, "#%s", asctime(localtime(&tp))); /* includes \n */
/*	cp = print_data_line(a, &cp);
	fprintf(f, "#data(%s): %s", name_identifier(a->id), cp);
	if (a != b) {
		cp = print_data_line(b, &cp);
		fprintf(f, " data(%s): %s", name_identifier(b->id), cp);
	}
	if (cp /*!= NULL*///)
/*		efree(cp);
	fprintf(f, "\n");
	fprintf(f, "#[1] mean: %g variance: %g", a->mean, a->std * a->std);
	if (a != b)
		fprintf(f, " [2] mean: %g variance: %g", b->mean, b->std * a->std);
	fprintf(f, "\n");
	fprintf(f, "#cutoff: %g ", ev->cutoff);
	if (!gl_bounds /*== NULL*///)
/*		fprintf(f,"%s %g\n","interval width:", ev->iwidth);
	else
		fprintf(f, "(fixed boundaries)\n");
	if (! ev->is_directional)
		fprintf(f, "#direction: total ");
	else
		fprintf(f,"#alpha <x,y>: %gd +/- %g; beta <alpha,z>: %gd +/- %g%s",
			gl_alpha, gl_tol_hor, gl_beta, gl_tol_ver,
			gl_sym_ev ? " (symmetric)" : "");
	fprintf(f, "\n");
	if (ev->cloud)
		fprintf(f, "# i  j distance %s cloud\n", vgm_type_str[ev->evt]);
	else
		fprintf(f, "#   from       to  n_pairs  av_dist %s\n",
			vgm_type_str[ev->evt]);
	return;
} */

static void fprint_header_vgm(CEFile* f,const DATA *a, const DATA *b,const SAMPLE_VGM *ev)
{
	extern char *command_line;
	char *cp = NULL;
    char buff[BUFF_SIZE];
    double tmpdouble;/*KS added*/


	sprintf(buff, "#gstat %s %s [%s]", GSTAT_OS, VERSION, command_line);
	f->fprintf(buff);
	sprintf(buff, "\n");
    f->fprintf(buff);
	sprintf(buff, "#sample %s%s\n",
		(ev->evt == CROSSVARIOGRAM && ev->pseudo ? "pseudo " : ""),
		vgm_type_str[ev->evt]);
     f->fprintf(buff);
	cp = print_data_line(a, &cp);
	sprintf(buff, "#points(%s): %s\n", name_identifier(a->id), cp);     /*KS added carriage return*/
    f->fprintf(buff);
	if (a != b)
    {
		cp = print_data_line(b, &cp);
		sprintf(buff, "#points(%s): %s\n", name_identifier(b->id), cp);  /*KS added # and carriage return*/
        f->fprintf(buff);
	}

	if (cp != NULL)
    {
		efree(cp);
    }
/*	sprintf(buff, "\n");
    f->fprintf(buff); *//*KS put above */
    tmpdouble = a->std * a->std;
	sprintf(buff, "#[1] mean: %g variance: %g", a->mean, tmpdouble);      /*KS added chnaged to tmpsouble*/
    f->fprintf(buff);
	if (a != b)
    {
        tmpdouble = b->std * a->std;
		sprintf(buff, " [2] mean: %g variance: %g", b->mean, tmpdouble);   /*KS added chnaged to tmpsouble*/
        f->fprintf(buff);
    }
	sprintf(buff, "\n");
    f->fprintf(buff);
	sprintf(buff, "#cutoff: %g ", ev->cutoff);
    f->fprintf(buff);
	if (!gl_bounds)
    {
		sprintf(buff,"%s %g\n","interval width:", ev->iwidth);
        f->fprintf(buff);
    }
	else
    {
		sprintf(buff, "(fixed boundaries)\n");
        f->fprintf(buff);
    }
	if (! ev->is_directional)
	{
		sprintf(buff, "#direction: total ");
        f->fprintf(buff);
    }
	else
    {
		sprintf(buff,"#alpha <x,y>: %gd +/- %g; beta <alpha,z>: %gd +/- %g%s",
			gl_alpha, gl_tol_hor, gl_beta, gl_tol_ver,
			gl_sym_ev ? " (symmetric)" : "");
        f->fprintf(buff);
    }
	sprintf(buff, "\n");
    f->fprintf(buff);
	if (ev->cloud)
    {
		sprintf(buff, "# i  j distance %s cloud\n", vgm_type_str[ev->evt]);
        f->fprintf(buff);
    }
	else
    {
		sprintf(buff, "#   from       to  n_pairs   avdist %s\n",
			vgm_type_str[ev->evt]);
        f->fprintf(buff);
    }
    f->commit();
	return;
}

static SAMPLE_VGM *alloc_exp_variogram(DATA *a, DATA *b, SAMPLE_VGM *ev) {
	int i;
	double nd;

	assert(a /*!= NULL*/);
	assert(ev /*!= NULL*/);

	if (gl_zero_est != ZERO_DEFAULT && ev->zero != gl_zero_est)
		ev->zero = gl_zero_est;

	if (gl_gls_residuals) {
		if (a->calc_residuals)
			make_gls(a, 1);
		if (b /*!= NULL*/ && b->calc_residuals)
			make_gls(b, 1);
	} else {
		if (a->calc_residuals)
			make_residuals_lm(a);
		if (b /*!= NULL*/ && b->calc_residuals)
			make_residuals_lm(b);
	}

	if (ev->cloud) {
		ev->n_est = 0;
		return ev;
	}

	if (gl_bounds /*!= NULL*/) {
		for (i = ev->n_est = 0; gl_bounds[i] >= 0.0; i++)
			ev->n_est++;
	} else {
/* check for overflow: */
		nd = floor(ev->cutoff / ev->iwidth) + 1;
		if (nd > INT_MAX) {
			pr_warning("choose a larger width or a smaller cutoff value");
			ErrMsg(ER_MEMORY, "(experimental variogram too large)");
		}
		ev->n_est = (int) nd;
	}
/*
 * zero est go to ev->gamma[ev->n_est - 1], ev->nh[ev->n_est - 1]
 */
	if (ev->zero)
		ev->n_est++;

	resize_ev(ev, ev->n_est);

/* initialize: */
	for (i = 0; i < ev->n_est; i++) {
		ev->gamma[i] = 0.0;
		ev->dist[i] = 0.0;
		ev->nh[i] = 0; 
		ev->pairs[i] = (DPOINT **) NULL;
	}
	return ev;
}

static void resize_ev(SAMPLE_VGM *ev, unsigned int size) {
	if (size > ev->n_max) {
		ev->n_max = size;
		ev->gamma = (double *) erealloc (ev->gamma, ev->n_max * sizeof(double));
		ev->dist = (double *) erealloc (ev->dist, ev->n_max * sizeof(double));
		ev->nh = (unsigned long *) erealloc (ev->nh, ev->n_max * sizeof(long));
		ev->pairs = (DPOINT ***)
				erealloc(ev->pairs, ev->n_max * sizeof(DPOINT **));
	}
}

static void *register_pairs(void *pairs, unsigned long nh,
		DPOINT *a, DPOINT *b) {
	/* resize pairs; add a and b to it */
	if (gl_register_pairs == 0)
		return NULL;
	if (nh % SEM_INCREMENT == 0)
		pairs = erealloc(pairs, 2 * (nh + SEM_INCREMENT + 1) * sizeof(DPOINT **));
	((DPOINT **) pairs)[2 * nh] = a;
	((DPOINT **) pairs)[2 * nh + 1] = b;
	return pairs;
}

static void push_to_cloud(SAMPLE_VGM *ev, double gamma, double dist,
	unsigned long index) {

	if (ev->n_est == ev->n_max)
		resize_ev(ev, ev->n_max + SEM_INCREMENT);

	ev->gamma[ev->n_est] = gamma;
	ev->dist[ev->n_est]  = dist;
	ev->nh[ev->n_est]    = index;
	ev->pairs[ev->n_est] = NULL;
	ev->n_est++;
}

SAMPLE_VGM * SetEVnest(DATA *a, DATA *b, SAMPLE_VGM *ev)
{

	int i;

	if (!a)
		ErrMsg(ER_NULL, "alloc_exp_variogram(): a == NULL");
	if (!ev)
		ErrMsg(ER_NULL, "alloc_exp_variogram(): ev == NULL");
	if (gl_zero_est != ZERO_DEFAULT && ev->zero != gl_zero_est)
		ev->zero = gl_zero_est;
	if (ev->cloud == 0)
    {
		if (gl_bounds != NULL)
        {
			for (i = ev->n_est = 0; gl_bounds[i] >= 0.0; i++)
			{
				ev->n_est++;
			}
		}
        else
			ev->n_est = ((int) floor(ev->cutoff / ev->iwidth)) + 1;
	}
    else
    {
		if (b != NULL)
			ev->n_est = a->n_list * b->n_list;
		else
			ev->n_est = (a->n_list * (a->n_list + 1)) / 2;
	}
	if (ev->zero)
		ev->n_est++; /* for last element */

	return ev;
}

SAMPLE_VGM * AllocVGM(DATA *a, DATA *b, SAMPLE_VGM *ev)
{
if (gl_zero_est != ZERO_DEFAULT && ev->zero != gl_zero_est)
		ev->zero = gl_zero_est;

	if (gl_gls_residuals) {
	/*	if (a->calc_residuals)
			make_gls_residuals(a);
		if (b != NULL && b->calc_residuals)
			make_gls_residuals(b);*/
	} else {
		if (a->calc_residuals)
			make_residuals_lm(a);
		if (b != NULL && b->calc_residuals)
			make_residuals_lm(b);
	}

	ev->gamma = (double *) erealloc (ev->gamma, sizeof(double) * ev->m_memsize);
	ev->dist = (double *) erealloc (ev->dist, sizeof(double) * ev->m_memsize);
    ev->nh = (unsigned long *) erealloc (ev->nh, sizeof(long) * ev->m_memsize);
 	ev->distx = (double *) erealloc (ev->distx, sizeof(double) * ev->m_memsize);
 	ev->disty = (double *) erealloc (ev->disty, sizeof(double) * ev->m_memsize);
	if((COVARIOGRAM == ev->evt) ||
		(CROSSCOVARIOGRAM == ev->evt))
	{
		ev->Xatt = (double*) erealloc( ev->Xatt, sizeof(double)* ev->m_memsize);
		ev->Yatt = (double*) erealloc( ev->Yatt, sizeof(double)* ev->m_memsize);
	}

    if (CORRELOGRAM)
    {
		ev->SQRXatt = (double*) erealloc( ev->SQRXatt, sizeof(double)* ev->m_memsize);
		ev->SQRYatt = (double*) erealloc( ev->SQRYatt, sizeof(double)* ev->m_memsize);
	}
	ResetEVspace( ev );

	return ev;
}
void ResetEVspace( SAMPLE_VGM *ev )
{
	int i;

    i = ev->m_memsize*sizeof(double);
    FillMemory(ev->gamma,i,0);
    FillMemory(ev->dist,i,0);
    FillMemory(ev->distx,i,0);
    FillMemory(ev->disty,i,0);
    if((COVARIOGRAM == ev->evt) ||
		(CROSSCOVARIOGRAM == ev->evt)) {
            FillMemory(ev->Xatt,i,0);
            FillMemory(ev->Yatt,i,0);
            }
    if (CORRELOGRAM) {
        FillMemory(ev->SQRXatt,i,0);
        FillMemory(ev->SQRYatt,i,0);
    }
    FillMemory(ev->nh,ev->m_memsize*sizeof(unsigned long),0);
}

void PageDivide(SAMPLE_VGM *ev)
{
	int i;

	if (ev->cloud)
		return; /* has been done in the first round */
	bool bLastPage = 1/*FALSE*/;
	int max ;
	int upper = ev->m_memsize * (1 + ev->m_pos);
	(upper >=  ev->n_est )? bLastPage = 0/*TRUE*/,max = ev->n_est: max = upper;

	for (i = 0; i < max; i++)
	{
		if (ev->nh[i])
		{
			ev->dist[i] /= ev->nh[i];
			switch (ev->evt)
			{
				case SEMIVARIOGRAM:
					if (gl_cressie)
						ev->gamma[i] = 0.5 * pow(ev->gamma[i]/ev->nh[i], 4.0)
							/(0.457 + 0.494 / ev->nh[i]);
					else
						ev->gamma[i] /= (2.0 * ev->nh[i]);
					break;
				case CROSSVARIOGRAM:
					ev->gamma[i] /= (2.0 * ev->nh[i]);
					break;
				case COVARIOGRAM: /* BREAKTHROUGH */
				case CROSSCOVARIOGRAM:

                        ev->gamma[i] /= (1.0 * ev->nh[i]);
//						ev->Xatt[i] /= ev->nh[i];
//            			ev->Yatt[i] /= ev->nh[i];
                        if (!resid)
                    	ev->gamma[i] -= ((ev->Xatt[i]/ev->nh[i]) * (ev->Yatt[i]/ev->nh[i]));

					break;
				case NOTSPECIFIED: /* BREAKTHROUGH */
				default:
					ErrMsg(ER_IMPOSVAL, "case ev->evt");
			}
            if ((CORRELOGRAM) && (!ev->cloud))
			{
						ev->SQRXatt[i] = /*SQR(ev->Xatt[i]) - SQR*/(ev->SQRXatt[i] / ev->nh[i]);
						ev->SQRYatt[i] = /*SQR(ev->Yatt[i]) - SQR*/(ev->SQRYatt[i] / ev->nh[i]);

                        if (ev->gamma[i] != 0.0)
                    		ev->gamma[i] /= sqrt(ev->SQRXatt[i] * ev->SQRYatt[i]);
            }
		}
	}
    /*KS*/
    if ((ev->evt == CROSSCOVARIOGRAM) || (ev->evt == COVARIOGRAM)) {
        if (ev->nh_last > 0)
    	{
         ev->gamma_last /= (1.0 * ev->nh_last);
         if (!resid) ev->gamma_last -=  ((ev->Xatt_n_est/ev->nh_last) * (ev->Yatt_n_est/ev->nh_last));
        }   			/*KS added*/
        }

    if ((CORRELOGRAM) && (!ev->cloud))
			{
            if (ev->nh_last > 0)
                 {
						ev->SQRXatt_n_est = /*SQR(ev->Xatt_n_est) - SQR*/(ev->SQRXatt_n_est / ev->nh_last);
						ev->SQRYatt_n_est = /*SQR(ev->Yatt_n_est) - SQR*/(ev->SQRYatt_n_est / ev->nh_last);
						//if (ev->gamma_last!=0)
                        if (ev->gamma_last != 0.0)
                    		ev->gamma_last /= sqrt(ev->SQRXatt_n_est * ev->SQRYatt_n_est);
                 }
            }
}

static int pos = 0;
void Pagefprint_sample_vgm(CEFile* f, const SAMPLE_VGM *ev, int n_list1, int n_list2)
{

	int i, n,index,inbound,outbound,printsize;
	float from, to, tmpfloat;
    char buff[BUFF_SIZE*2];
    #define EVFMT "%8g %10.7g %8ld %8g %15.15g\n"

	from = 0.0; to = 0.0;
	inbound  = ev->m_pos * ev->m_memsize;
	outbound = (inbound + ev->m_memsize > ev->n_est) ?  ev->n_est: inbound + ev->m_memsize;
	index = ev->n_est % ev->m_memsize;
    if( 0 == index)
    {
    	if( outbound == ev->n_est)
        {
            index = ev->n_est;
        }
        else
        {
         	index = ev->m_memsize;
        }
    }
	if (! ev->cloud)
    {
		if( 0 == ev->m_pos )
		{
			if (ev->zero == ZERO_SPECIAL || ev->zero == ZERO_AVOID)
			{
				n = ev->n_est-1;
			}
			else
			{
				n = ev->n_est;
			}
			sprintf(buff,"# %d\n", n);
			f->fprintf(buff);
			/* start writing: distance 0 */
   			if ( (ev->zero == ZERO_SPECIAL)&& (ev->nh_last) )
			{
				sprintf(buff, EVFMT, 0.0, 0.0, ev->nh_last,
					ev->dist_last, ev->gamma_last);
				f->fprintf(buff);
			}
		}
		/* continue writing: */
		if( ev->n_est == outbound)
		{
        	//this will work if ev->m_memsize < ev->n_est because in that
            //case ev->n_est = outbound
			if (ev->zero == ZERO_SPECIAL || ev->zero == ZERO_AVOID)
			{
                n = index -1;
			}
			else
			{
				//n = ev->n_est;
				n = index;
			}
        }
		else
		{
			n = ev->m_memsize;
		}

        for (i = 0; i < n; i++)
        {
			if (ev->nh[i] > 0)
            {
				if (!gl_bounds)
                {
				  	from =( (ev->m_memsize* pos) + i)*ev->iwidth;
					to = ((ev->m_memsize* pos) +i+1)*ev->iwidth;
				}
				else
                {
					if (i == 0)
                    {
						from = 0.0;
                    }
                	else
                	{
						from = gl_bounds[i-1];
                    }
					to = gl_bounds[i];
				}
				to = MIN(ev->cutoff, to);
				sprintf(buff, EVFMT, from, to, ev->nh[i],
					ev->dist[i], ev->gamma[i]);
                f->fprintf(buff);
			}
		}
	}
    else
    {
		if(0 == ev->m_pos )
		{
		}
        if(  outbound != ev->n_est)
		{

			index = ev->m_memsize;
		}
        for (i = 0; i < index; i++)
        {
            f->fwrite(&ev->distx[i],8,1);
            f->fwrite(&ev->disty[i],8,1);
            f->fwrite(&ev->gamma[i],8,1);
        }  /*KS added*/
	}
	if (! ev->cloud) f->commit();
    	else f->commit2();
	return;
} /* fprint_sample_vgm */

static int get_index(double dist, float iwidth /*SAMPLE_VGM *ev*/) {
	double frac;
	int i = 0;

	/*if (dist == 0.0 && ev->zero != ZERO_INCLUDE)
		return ev->n_est - 1;
    *//*KS removed*/
	if (gl_bounds != DEF_bounds) {
		for (i = 0; gl_bounds[i] >= 0.0; i++)
			if (dist <= gl_bounds[i])
				return i;
		assert(0);
	}

	assert(/*ev->*/iwidth > 0.0);

	frac = dist / /*ev->*/iwidth;
	if (dist > 0.0 && frac == floor(frac))
		return (int) (floor(frac)) - 1;
	else
		return (int) floor(frac);
}

static int GetIndex(DATA *d, SAMPLE_VGM *ev)
{
	int i, j, index = 0;
	double ddist;
	for (i = 0; i < d->n_list; i++)
	{
		for (j = 0; j < (ev->map != NULL ? d->n_list : i); j++)
		{
			ddist = valid_distance(d->list[i], d->list[j], ev->cutoff, 1,
				d, d, (GRIDMAP *) ev->map);
			if (ddist >= 0.0 && i != j)
			{
                 index++;
                 ev->gamma[j] = 0.0;
			}/* if ddist >= 0 */
		}  /* for j */
	} /* for i */
   return index;
}

static void divide(SAMPLE_VGM *ev) {
	int i;

	if (ev->cloud)
		return; /* has been done in the first round */
	for (i = 0; i < ev->n_est; i++) {
		if (ev->nh[i]) {
			ev->dist[i] /= ev->nh[i];
			switch (ev->evt) {
				case SEMIVARIOGRAM:
					if (gl_cressie)
						ev->gamma[i] = 0.5 * pow(ev->gamma[i]/ev->nh[i], 4.0)
							/(0.457 + 0.494 / ev->nh[i]);
					else
						ev->gamma[i] /= (2.0 * ev->nh[i]);
					break;
				case CROSSVARIOGRAM:
					ev->gamma[i] /= (2.0 * ev->nh[i]);
					break;
				case COVARIOGRAM: /* BREAKTHROUGH */
				case CROSSCOVARIOGRAM:
					ev->gamma[i] /= (1.0 * ev->nh[i]);
					break;
				case NOTSPECIFIED: /* BREAKTHROUGH */
				default:
					assert(0);
					break;
			}
		}
	}
}

/*void fprint_sample_vgm(FILE *f, const SAMPLE_VGM *ev) {
#define EVFMT "%8g %8g %8ld %8g %8g\n"
	int i, n;
	double from, to;

	if (! ev->cloud) {
		/* start writing: distance 0 */
/*		if (ev->zero == ZERO_SPECIAL && ev->nh[ev->n_est-1])
			fprintf(f, EVFMT, 0.0, 0.0, ev->nh[ev->n_est-1],
				ev->dist[ev->n_est-1], ev->gamma[ev->n_est-1]);
		/* continue writing: */
/*		if (ev->zero == ZERO_SPECIAL || ev->zero == ZERO_AVOID)
			n = ev->n_est - 1;
		else
			n = ev->n_est;
		for (i = 0; i < n; i++) {
			if (ev->nh[i] > 0) {
				if (!gl_bounds /*== NULL*///) {
/*					from = i*ev->iwidth;
					to = (i+1)*ev->iwidth;
				} else {
					if (i == 0)
						from = 0.0;
					else
						from = gl_bounds[i-1];
					to = gl_bounds[i];
				}
				to = MIN(ev->cutoff, to);
				fprintf(f, EVFMT, from, to, ev->nh[i],
					ev->dist[i], ev->gamma[i]);
				/*
				for (j = 0; j < ev->nh[i]; j++)
					fprintf(f, "[%d,%d] ",
						GET_INDEX(((DPOINT ***)ev->pairs)[i][2*j]),
						GET_INDEX(((DPOINT ***)ev->pairs)[i][2*j+1]));
				fprintf(f, "\n");
				*/
/*			}
		}
	} else {
		for (i = 0; i < ev->n_est; i++)
			fprintf(f, "%ld %ld %g %g\n", HIGH_NH(ev->nh[i]) + 1,
				LOW_NH(ev->nh[i]) + 1, ev->dist[i], ev->gamma[i]);
	}
	fflush(f);
	return;
} /* fprint_sample_vgm */

static void fprint_sample_vgm(CEFile* f, const SAMPLE_VGM *ev, int n_list1, int n_list2)
{
/*KS #define EVFMT "%8g %8g %8ld %8g %8g\n" KS replaced*/
//#define EVFMT "%8g %10.7g %8ld %8g %15.15g\n"
	int i, n;
	float from, to;
    char buff[BUFF_SIZE];
	from = 0.0; to = 0.0;
	if (! ev->cloud)
    {
		if (ev->zero == ZERO_SPECIAL || ev->zero == ZERO_AVOID)
        //{
			n = ev->n_est-1;
        //}
		else
        //{
			n = ev->n_est;
        //}
        sprintf(buff,"# %d\n", n);
        f->fprintf(buff);

		/* start writing: distance 0 */
   		if ( (ev->zero == ZERO_SPECIAL)&& (ev->nh[ev->n_est-1]) )
        {
			sprintf(buff, EVFMT, 0.0, 0.0, ev->nh[ev->n_est-1],
				ev->dist[ev->n_est-1], ev->gamma[ev->n_est-1]);
            f->fprintf(buff);
        }
		/* continue writing: */
		if (ev->zero == ZERO_SPECIAL || ev->zero == ZERO_AVOID)
        //{
			n = ev->n_est - 1;
        //}
		else
        //{
			n = ev->n_est;
        //}

        for (i = 0; i < n; i++)
        {
			if (ev->nh[i] > 0)
            {
				if (!gl_bounds)
                {
				  	from = i*ev->iwidth;
					to = (i+1)*ev->iwidth;
				} else
                {
					if (i == 0)
                    //{
						from = 0.0;
                    //}
                	else
                	//{
						from = gl_bounds[i-1];
                    //}
					to = gl_bounds[i];
				}
				to = MIN(ev->cutoff, to);
				sprintf(buff, EVFMT, from, to, ev->nh[i],
					ev->dist[i], ev->gamma[i]);
                f->fprintf(buff);
			}
		}
	}
    else
    {
	/*	for (i = 0; i < ev->n_est; i++)
			fprintf(f, "%ld %ld %g %g\n", HIGH_NH(ev->nh[i]) + 1,
				LOW_NH(ev->nh[i]) + 1, ev->dist[i], ev->gamma[i]); */
        i = ev->n_est-1;
        sprintf(buff, "%d\n", i);
        f->fprintf(buff);
        for (i = 0; i < ev->n_est; i++)
        {
            sprintf(buff, "%f %f %f\n", ev->distx[i], ev->disty[i], ev->gamma[i]);// ev->dist[i]);
            f->fprintf(buff);
        }  /*KS added*/
	}
	f->commit();
	return;
} /* fprint_sample_vgm */
static void ev2map(VARIOGRAM *v) {
	GRIDMAP *m1 = NULL, *m2 = NULL;
	unsigned int row, col, i;
	SAMPLE_VGM *ev;

	if (!(v->fname) /*== NULL*/)
		return;
	ev = v->ev;
	m1 = (gridmap *) map_dup(v->fname, (gridmap *) ev->map);
	if (v->fname2 /*!= NULL*/)
		m2 = (gridmap *)map_dup(v->fname2, (gridmap *) ev->map);
	for (row = i = 0; row < m1->rows; row++) {
		for (col = 0; col < m1->cols; col++) {
			if (ev->nh[i] > 0)
				map_put_cell(m1, row, col, ev->gamma[i]);
			if (m2 /*!= NULL*/)
				map_put_cell(m2, row, col, 1.0 * ev->nh[i]);
			i++;
		}
	}
	m1->write(m1);
	if (m2 /*!= NULL*/)
		m2->write(m2);
	return;
}

static SAMPLE_VGM *load_ev(SAMPLE_VGM *ev, const char *fname) {
	char *s = NULL, *tok;
	int i, size = 0, incr = 100;
	unsigned long l;
	FILE *f;

	f = efopen(fname, "r");
	if (!ev /*== NULL*/)
		ev = init_ev();
	ev->evt = SEMIVARIOGRAM;
	for (i = 1; i <= 8; i++) {
		get_line(&s, &size, f);
		if (i == 6) {
			tok = strtok(s, " "); /* word */
			tok = strtok(NULL, " "); /* cutoff */
			if (read_double(tok, &(ev->cutoff))) {
				fclose(f); efree(s);
				/*pr_warning*/sprintf(gerr_text,"file: %s, line: %d, token: %s", fname, i, tok);
                pr_warning(gerr_text);
				return NULL;
			}
			tok = strtok(NULL, " "); /* word */
			tok = strtok(NULL, " "); /* word */
			tok = strtok(NULL, " \n"); /* iwidth */
			if (read_double(tok, &(ev->iwidth))) {
				fclose(f); efree(s);
				/*pr_warning*/sprintf(gerr_text,"file: %s, line: %d, token: %s", fname, i, tok);
                pr_warning(gerr_text);
				return NULL;
			}
		}
	}
	while (get_line(&s, &size, f) != NULL) {
		ev->n_est++;
		if (ev->n_est >= ev->n_max) {
			ev->n_max += incr;
			ev->gamma = (double *) erealloc
				(ev->gamma, sizeof(double) * ev->n_max);
			ev->dist = (double *) erealloc
				(ev->dist, sizeof(double) * ev->n_max);
			ev->nh = (unsigned long *) erealloc
				(ev->nh, sizeof(long) * ev->n_max);
		}
		tok = strtok(s, " "); /* from */
		tok = strtok(NULL, " "); /* to */
		tok = strtok(NULL, " "); /* nh */
		if (read_ulong(tok, &l)) {
			fclose(f); efree(s);
			/*pr_warning*/sprintf(gerr_text,"file: %s, line: %d, token: %s", fname, ev->n_est+8, tok);
            pr_warning(gerr_text);
			return NULL;
		}
		ev->nh[ev->n_est-1] = l;
		tok = strtok(NULL, " "); /* dist */
		if (read_double(tok, &(ev->dist[ev->n_est-1]))) {
			fclose(f); efree(s);
			/*pr_warning*/sprintf(gerr_text,"file: %s, line: %d, token: %s", fname, ev->n_est+8, tok);
            pr_warning(gerr_text);
			return NULL;
		}
		tok = strtok(NULL, " \n"); /* semivariance or whatever */
		if (read_double(tok, &(ev->gamma[ev->n_est-1]))) {
			fclose(f); efree(s);
			/*pr_warning*/sprintf(gerr_text,"file: %s, line: %d, token: %s", fname, ev->n_est+8, tok);
            pr_warning(gerr_text);
			return NULL;
		}
	}
	efree(s);
	efclose(f);
	return ev;
}
