/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/

/*
 * read.c: read int or float from a string, missing value and error handling
 */
#include <stdio.h>
#include <stdlib.h>
#include <sysutils.hpp>
#include <errno.h>
#include <float.h>
#include <limits.h>
#include <math.h>
#include <string.h>

#include "defs.h"
#include "utils.h"
#include "userio.h"
#include "glvars.h"
#include "read.h"
#include "gstat_main.h"

/*
 * functions for save reading and error checking
 * on reading int, long, float and double
 */

#ifndef INT_MAX
#	define INT_MAX 32767
#	define INT_MIN -32767
#endif
#ifndef LONG_MAX
#	define LONG_MAX +2147483647L
#	define LONG_MIN -2147483647L
#endif
#ifndef FLT_MAX
#	define FLT_MAX 1.0E+37F
#	define FLT_MIN 1.0E-37F
#endif
#ifndef HUGE_VAL
#  define HUGE_VAL    1.7976931348623157e+308
#endif

extern char gerr_text[ERR_TEXT_SIZE];
//#ifdef MAIN_READ KS switched to interior for compile
int main(int argc, char *argv[]) {
	float f;
	double d;
	int i, c;
	unsigned int u;
	extern char *optarg;
	extern int optind, optopt, opterr, getopt(int, char **, const char *);
#ifdef MAIN_READ
	opterr = 0;
	while ((c = getopt(argc, argv, "i:f:d:")) != EOF) {
		switch (c) {
			case 'u':
				if (read_uint(optarg, &u))
					printf("error reading uint from %s\n", optarg);
				else
					printf("unsigned int '%s' value %u\n", optarg, u);
				break;
			case 'i':
				if (read_int(optarg, &i))
					printf("error reading int from %s\n", optarg);
				else
					printf("int '%s' value %d\n", optarg, i);
				break;
			case 'f':
				if (read_float(optarg, &f))
					printf("error reading float from %s\n", optarg);
				else
					printf("float '%s' value %g\n", optarg, f);
				break;
			case 'd':
				if (read_double(optarg, &d))
					printf("error reading float from %s\n", optarg);
				else
					printf("double '%s' value %g\n", optarg, d);
				break;
			case '?':
				ErrClo(optopt);
			default:
				printf("%s -i [int] -f [float] -d [double]", argv[0]);
		}
	}
    #endif
}
//#endif

int read_float(const char *s, float *f) {
/* return 1 on error, 0 on no error */
	double d = 0, min3 = (double) FLT_MIN, max3 = (double) FLT_MAX;/*KS*/
	int warning = 0;

	warning = read_double(s, &d);
	if (is_mv_double(&d)) {
		set_mv_float(f);
		return 0;
	}
	if (fabs(d) > max3 || (fabs(d) < min3 && d != 0.0)) {  /*KS*/
		message("value outside valid 4-byte range" /*+/-[%g, %g]\n", min, max*/);
		ErrMsg(ER_RANGE, s);
	}
    //TextToFloat(const_cast <char *>(s), f, fvExtended);/*KS replaced*/
	*f = (float) d;
	return warning;
}

int read_double(const char *s, double *d) {
/* return 1 on error, 0 on no error */
    int ch ='.';

	char *cp2 =NULL;             /*KS changed from cp to cp2*/
    long value;

	if (!s /*== NULL*/)
		ErrMsg(ER_NULL, "read_double()");
	if (s[0] == '\0')
		ErrMsg(ER_IMPOSVAL, "read_double(): empty string");
	if (strcmp(s, const_cast <const char *> (gl_mv_string)) == 0) {
		set_mv_double(d);
		return 0;
	}
    /*cp2 = strchr(const_cast <char *> (s), ch );
    if( !cp2 )// == NULL )
    {
       sscanf(s,"%d",&value);
       *d = value;
    }
    else
    {
      sscanf(s,"%lf",d);
    } */
	*d = strtod(s, &cp2);
//    TextToFloat(const_cast <char *>(s), d, fvExtended);/*KS replaced*/
	if (errno == ERANGE) {
		message("value outside valid range +/-[%g, %g]\n", DBL_MIN, DBL_MAX);
		ErrMsg(ER_RANGE, s);
	}
//    return 0;
	if (*cp2 == '\0')
		return 0;
	else {
#ifdef READ_WARNING
		/*pr_warning*/sprintf(gerr_text,"read_double(): unconverted suffix: `%s'", cp2);
        pr_warning(gerr_text);
#endif
		return 1;
	}
}

int read_int(const char *s, int *i) {
/* return 1 on error, 0 on no error */
	long l;
	int warning = 0;

	warning = read_long(s, &l);
	if (warning) 
		return warning;
	if (l > INT_MAX || l < INT_MIN) {
		message("value outside valid range [%d, %d]\n", INT_MIN, INT_MAX);
		ErrMsg(ER_RANGE, s);
	}
	*i = (int) l;
	return warning;
}

int read_uint(const char *s, unsigned int *u) {
/* return 1 on error, 0 on no error */
	unsigned long ul;
	int warning = 0;

	warning = read_ulong(s, &ul);
	if (warning) 
		return warning;
	if (ul > UINT_MAX) {
		message("value outside valid range [0, %u]\n", UINT_MAX);
		ErrMsg(ER_RANGE, s);
	}
	*u = (unsigned int) ul;
	return warning;
}

int read_long(const char *s, long *l) {
/* return 1 on error, 0 on no error */
	char *cp;

	if (!s /*== NULL*/)
		ErrMsg(ER_NULL, "read_long()");
	if (s[0] == '\0') 
		ErrMsg(ER_IMPOSVAL, "read_long(): empty string");
	*l = strtol(s, &cp, 10);
	if (errno == ERANGE) {
		message("value outside valid range [%ld, %ld]\n", LONG_MIN, LONG_MAX);
		ErrMsg(ER_RANGE, s);
	}
	if (*cp == '\0')
		return 0;
	else  {
#ifdef READ_WARNING
		/*pr_warning*/sprintf(gerr_text,"read_long(): unconverted suffix: `%s'", cp);
        pr_warning(gerr_text);
#endif
		return 1;
	}
}

int read_ulong(const char *s, unsigned long *u) {
/* return 1 on error, 0 on no error */
	char *cp;

	if (!s /*== NULL*/)
		ErrMsg(ER_NULL, "read_long()");
	if (s[0] == '\0') 
		ErrMsg(ER_IMPOSVAL, "read_long(): empty string");
	*u = strtoul(s, &cp, 10);
	if (errno == ERANGE) {
		message("value outside valid range [0, %lu]\n", ULONG_MAX);
		ErrMsg(ER_RANGE, s);
	}
	if (*cp == '\0')
		return 0;
	else  {
#ifdef READ_WARNING
		/*pr_warning*/sprintf(gerr_text,"read_long(): unconverted suffix: `%s'", cp);
        pr_warning(gerr_text);
#endif
		return 1;
	}
}
