%{
/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/

/*
 * parse.y: LALR(1) grammar for the gstat command syntax.
 * to make parse.c, type ``make parse.c'', it will use bison or yacc.
 *
 * If you fail (or don't have bison or yacc), then copy the file parse.c_
 * to parse.c. All this can be prevented by running configure while NO_YACC
 * is defined.
 * 
 * The parser assumes that in the function yylex() each identifier is
 * duplicated to ylval.sval, not just a pointer-copy. (some memory loss
 * will occur as a result)
 *
 * hints to extend the parser: 
 * o add a command: copy all from the most similar available command
 *   (add a %token and %type declaration, add a rule, add a return value
 *   from yylex() -> see the IDENT actions in lex.l)
 * o add a ``set'' variable: modify is_set_expr(), glvars.[ch] and defaults.h
 * o add a data() command: modify data.[ch] and is_data_expr()
 * o add a variogram model: vario*.[ch]
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <io.h>

#include "defs.h"




#include "data.h"
#include "vario.h"
#include "debug.h"
#include "glvars.h"
#include "userio.h"
#include "utils.h"
#include "lex.h"

static DATA *d = NULL, **dpp = NULL;
static DPOINT *bp = NULL;
static VARIOGRAM *v = NULL;
static int id = -1, id1 = -1, id2 = -1, col1 = -1, col2 = -1,
	fit_sill = 0, fit_range = 0, vector_only = 0, allow_vector_only = 0;
static double range = -1.0, anis[5];
static char **ofn = NULL, *boundary_file = NULL;
static VARIOGRAM *parse_variogram = NULL;
static D_VECTOR *sd_vector = NULL;

#ifdef YYBISON
# ifndef __STDC__
#  define __STDC__
/* or else all const's will be defined empty */
# endif
#endif

typedef struct {
	const char *name;
	void *ptr;
	enum { 
		UNKNOWN, 
		IS_INT, 
		IS_UINT, 
		IS_REAL, 
		IS_STRING, 
		IS_D_VECTOR, 
		NO_ARG 
	} what;
	enum { 
		NOLIMIT,
		GEZERO, 
		GTZERO
	} limit;
} GSTAT_EXPR;

GSTAT_EXPR expr = { NULL, NULL, 0/*UNKNOWN*/, 0/*NOLIMIT*/ };

static void push_to_v(const char *mod, double sill, double range, double *d,
		int fit_sill, int fit_range);
static void push_data_X(DATA *d, int id);
static ANIS_TM *get_tm(double anis[5]);
static int is_data_expr(DATA *d, GSTAT_EXPR *expr, const char *fld);
static int is_set_expr(GSTAT_EXPR *expr, const char *fld);
static int is_block_expr(GSTAT_EXPR *expr, const char *s);
static void push_marginal(char *name, double val);
static void check_assign_expr(GSTAT_EXPR *expr);
static void reset_parser(void);
static void verify_data(DATA *d);

#define yyerror(s) lex_error()

%}

%union {
	int ival;
	unsigned int uval;
	double dval;
	char *sval;
}

%token <ival> INT
%token <uval> UINT
%token <dval> REAL
%token <sval> QSTR  IDENT ID_DATA ID_X ID_VARIOGRAM ID_PREDICTIONS
%token <sval> ID_VARIANCES ID_COVARIANCES ID_OUTPUT ID_MASKS ID_EDGES ID_SET
%token <sval> ID_MERGE ID_AREA ID_BLOCK ID_METHOD ID_BOUNDS ID_MARGINALS

%type <sval> input assign any_id comcol command data_cmd data_decl 
%type <sval> data_cont data_exp data_what data_X data_X_what
%type <sval> vgm_cmd vgm_decl vgm_cont vgm_model vgm_model_type vgm_range
%type <sval> merge_cmd set_cmd set_exp set_lhs mask_cmd mask_cont
%type <sval> edges_cmd edges_cont block_cmd block_cont block_exp block_lhs
%type <sval> output_cmd method_cmd bounds_cmd bounds_exp 
%type <sval> marginals_cmd marginals_cont
%type <dval> val sill_val range_val d_vector d_list d_val

%%
input: { ; } 						    /* allow empty input */
	| input command { reset_parser(); }	/* or a list of commands */
	| d_list { vector_only = 1; }       /* or a list of numbers (stat.c) */
	;

command: ';' { ; }
	| data_cmd    ';' { ; }
	| vgm_cmd     ';' { update_variogram(v); }
	| merge_cmd   ';' { ; }
	| mask_cmd    ';' { ; }
	| edges_cmd  ';'  { ; }
    | block_cmd   ';' { ; }
	| area_cmd    ';' { ; }
	| output_cmd  ';' { ; }
	| method_cmd  ';' { ; }
	| bounds_cmd  ';' { ; }
	| marginals_cmd ';' { ; }
	| set_cmd     ';' { ; }
	;

val: INT { $$ = (double) $1; }
	| REAL ;

assign: '=' { ; }
	| ':' { ; }
	;

comcol: ':' { ; }
	| ',' { ; }
	;

d_vector: '[' d_list ']' { ; }

d_list: d_val 
	| d_list ',' d_val 
	| d_list d_val

d_val: val { 
			if (d == NULL)
				sd_vector = push_to_vector($1, sd_vector);
			else
				d->beta = push_to_vector($1, d->beta);
		}

any_id: IDENT | ID_AREA | ID_BLOCK | ID_BOUNDS | ID_COVARIANCES |
	ID_DATA | ID_MARGINALS | ID_MASKS | ID_METHOD | ID_OUTPUT |
	ID_PREDICTIONS | ID_SET | ID_VARIANCES | ID_VARIOGRAM | ID_X |
	ID_EDGES { ; }
	; /* allows things like  data(data) : ... ; etc. */

data_cmd: data_decl ':' data_cont { verify_data(d); } /* declaration : contents */
	| data_decl { d->dummy = 1; }
	; 

data_decl: ID_DATA '(' any_id ')' {
			id = which_identifier($3);
			dpp = get_data();
			d = dpp[id];
			d->id = id;
		}
	| ID_DATA '(' ')' {
			d = get_dataval();
			d->id = ID_OF_VALDATA;
		}
	| ID_DATA '(' error { ErrMsg(ER_SYNTAX, "invalid identifier"); }
	;

data_cont: data_exp				/* one data expression */
	| data_cont ',' data_exp	/* a list of data expressions */
	;

data_exp: { ; } /* can be empty */
	| ID_X '=' data_X
	| QSTR { d->fname = $1; }
	| data_what '=' INT {
			switch (expr.what) { 
				case 1/*IS_INT*/: *((int *)expr.ptr) = $3; break;
				case 3/*IS_REAL*/: *((double *)expr.ptr) = (double) $3; break;
				default: lex_error(); YYERROR; break;
			}
			check_assign_expr(&expr);
		}
	| data_what '=' REAL {
			if (expr.what != 3/*IS_REAL*/) {
				lex_error();
				YYERROR;
			}
			*((double *)expr.ptr) = $3;
			check_assign_expr(&expr);
		}
	| data_what '=' QSTR {
			if (expr.what != 4/*IS_STRING*/) {
				lex_error();
				YYERROR;
			}
			*((char **)expr.ptr) = $3;
		}
	| data_what '=' d_vector {
			if (expr.what != 5/*IS_D_VECTOR*/) {
				lex_error();
				YYERROR;
			}
			/*
			*((D_VECTOR **)expr.ptr) = sd_vector;
			printf("[[ %d ]]\n", sd_vector->size);
			sd_vector = NULL;
			*/
		}
	| data_what {
			if (expr.what != 6/*NO_ARG*/) {
				lex_error();
				YYERROR;
			}
		}
	;

data_X: data_X_what
	| data_X '&' data_X_what
	;

data_X_what: IDENT {
			for (id = 0; id < N_POLY; id++) {
				if (almost_equals($1, polynomial[id].name)) {
					id += POLY_MIN;
					break; /* i-loop */
				}
			}
			if (id < 0)
				data_add_X(d, id);
			else {
				lex_error();
				YYERROR;
			}
		}
	| INT { push_data_X(d, $1); }
	;

data_what: IDENT {
			if (! is_data_expr(d, &expr, $1)) {
				lex_error();
				YYERROR;
			}
		}
	;

block_cmd: ID_BLOCK {
			bp = get_block_p();
			bp->x = -1.0; /* will be set to grid cell size in predict.c */
		}
	| ID_BLOCK ':' block_cont
	;

block_cont: block_exp
	| block_cont ',' block_exp
	;

block_exp: block_lhs '=' val {
			*((double *)expr.ptr) = $3;
			check_assign_expr(&expr);
		}
	;

block_lhs: IDENT { if (! is_block_expr(&expr, $1)) { lex_error(); YYERROR; }}
	;

area_cmd: area_decl ':' data_cont { ; }
	;

area_decl: ID_AREA {
			create_data_area();
			d = get_data_area();
			d->id = ID_OF_AREA;
		}
	| ID_AREA '(' ')' {
			create_data_area();
			d = get_data_area();
			d->id = ID_OF_AREA;
		}
	;

vgm_cmd: vgm_decl ':' QSTR comcol vgm_cont { v->fname = $3; }
	| vgm_decl ':' QSTR comcol QSTR comcol vgm_cont { v->fname = $3; v->fname2 = $5; }
	| vgm_decl ':' QSTR {v->fname = $3; }
	| vgm_decl ':' QSTR comcol QSTR {v->fname = $3; v->fname2 = $5; }
	| vgm_decl ':' vgm_cont
	| vgm_decl ':' error ';' {
			/* this will eat the ';' as well, but we're bailing out anyway: */
			YYERROR;
		}
	;

vgm_decl: ID_VARIOGRAM '(' ')' {
			/* only allow this when called through read_variogram(): */
			assert(parse_variogram != NULL);
			v = parse_variogram;
			v->n_models = v->n_fit = 0;
		}
	| ID_VARIOGRAM '(' any_id ')' {
			id = which_identifier($3);
			v = get_vgm(LTI(id,id));
			v->id = v->id1 = v->id2 = id;
			v->n_models = v->n_fit = 0;
		}
	| ID_VARIOGRAM '(' any_id ',' any_id ')' {
			id1 = which_identifier($3);
			id2 = which_identifier($5);
			id = LTI(id1,id2);
			v = get_vgm(id);
			v->id = id;
			v->id1 = id1;
			v->id2 = id2;
			v->n_models = v->n_fit = 0;
		}
	;

vgm_cont: vgm_model
	| vgm_cont vgm_model
	;

vgm_model: sill_val vgm_model_type '(' ')' {
			push_to_v($2, $1, 0.0, NULL, fit_sill, fit_range);
		}
	| sill_val vgm_model_type '(' vgm_range ')' {
			push_to_v($2, $1, range, anis, fit_sill, fit_range);
		}
	| '+' sill_val vgm_model_type '(' vgm_range ')' {
			push_to_v($3, $2, range, anis, fit_sill, fit_range);
		}
	| '-' sill_val vgm_model_type '(' vgm_range ')' {
			push_to_v($3, -1.0 * $2, range, anis, fit_sill, fit_range);
		}
	;

vgm_model_type:	IDENT { 
			if (which_variogram_model($1) == NOT_SP) {
				lex_error(); YYERROR;
			}
	}

vgm_range: range_val { range = $1; anis[0] = -9999.0; }
	| range_val ',' val ',' val {
			range = $1;
			anis[0] = $3;
			anis[3] = $5;
			anis[1] = anis[2] = 0.0;
			anis[4] = 1.0;
		}
	| range_val ',' val ',' val ',' val ',' val ',' val {
			range = $1;
			anis[0] = $3;
			anis[1] = $5;
			anis[2] = $7;
			anis[3] = $9;
			anis[4] = $11;
		}
	;

sill_val: val { fit_sill = 1; }
	| '@' val { fit_sill = 0; $$ = $2; }
	| val '@' { fit_sill = 0; $$ = $1; }

range_val: val { fit_range = 1; }
	| '@' val { fit_range = 0; $$ = $2; }

output_cmd: ID_OUTPUT '=' QSTR { o_filename = $3; }
	| ID_PREDICTIONS '(' any_id ')' ':' QSTR { 
			id = which_identifier($3);
			ofn = (char **) get_outfile_name();
			ofn[2 * id] = $6;
		}
	| ID_PREDICTIONS ':' QSTR { 
			if (get_n_vars() == 0) {
				lex_error();
				ErrMsg(ER_SYNTAX, "define data first");
			}
			ofn = (char **) get_outfile_name();
			ofn[0] = $3;
		}
	| ID_VARIANCES '(' any_id ')' ':' QSTR { 
			id = which_identifier($3);
			ofn = (char **) get_outfile_name();
			ofn[2 * id + 1] = $6;
		}
	| ID_VARIANCES ':' QSTR { 
			if (get_n_vars() == 0) {
				lex_error();
				ErrMsg(ER_SYNTAX, "define data first");
			}
			ofn = (char **) get_outfile_name();
			ofn[1] = $3;
		}
	| ID_COVARIANCES '(' any_id ',' any_id ')' ':'  QSTR { 
			id = get_n_vars();
			id1 = which_identifier($3);
			id2 = which_identifier($5);
			if (id != get_n_vars())	
				ErrMsg(ER_SYNTAX, "define all data(..) before covariances(..,..)");
			ofn = (char **) get_outfile_name();
			id = 2 * id + LTI2(id1, id2);
			ofn[id] = $8;
		}
	;

set_cmd: ID_SET set_exp /* e.g. set nsim = 1 _or_ set out: "out" */
	| set_exp           /* e.g. nsim = 1  or  plot : 'file.plt" */
	; 

set_exp: set_lhs assign INT {
			switch (expr.what) { 
				case 1/*IS_INT*/: *((int *)expr.ptr) = $3; break;
				case 3/*IS_REAL*/: *((double *)expr.ptr) = (double) $3; break;
				default: lex_error(); YYERROR;
			}
			check_assign_expr(&expr);
		}
	| set_lhs assign UINT {
			switch (expr.what) {
				case 2/*IS_UINT*/: *((unsigned int *)expr.ptr) = $3; break;
				default: lex_error(); YYERROR;
			}
			check_assign_expr(&expr);
		}
	| set_lhs assign REAL {
			switch (expr.what) {
				case IS_REAL: *((double *)expr.ptr) = $3; break;
				default: lex_error(); YYERROR; break;
			}
			check_assign_expr(&expr);
		}
	| set_lhs assign QSTR {
			if (expr.what != 4/*IS_STRING*/) {
				lex_error();
				YYERROR;
			}
			*((char **) expr.ptr) = $3;
		}
	;

set_lhs: IDENT { if (! is_set_expr(&expr, $1)) { lex_error(); YYERROR; }}
	;

method_cmd: ID_METHOD ':' IDENT {
			for (id = 1; methods[id].name != NULL; id++) {
				if (almost_equals($3, methods[id].name)) {
					set_method(methods[id].m);
					break; /* id-loop */
				}
			}
			if (methods[id].m == NSP) {
				lex_error();
				YYERROR;
			}
		}
	;

mask_cmd: ID_MASKS ':' mask_cont
	;

mask_cont: QSTR { push_mask_name($1); }
	| mask_cont ',' QSTR { push_mask_name($3); }
	;

edges_cmd: ID_EDGES ':' edges_cont
	;

edges_cont: QSTR { push_edges_name($1); }
	| edges_cont ',' QSTR { push_edges_name($3); }
	;

merge_cmd: ID_MERGE any_id IDENT any_id {
			if (!almost_equals($3, "w$ith"))
				lex_error();
			id1 = which_identifier($2);
			id2 = which_identifier($4);
			col1 = col2 = 0;
			dpp = get_data();
			if (id1 < id2) { /* swap id's */
				id = id1; id1 = id2; id2 = id;
			}
			if (push_to_merge_table(dpp[id1], id2, col1, col2)) {
				lex_error();
				ErrMsg(ER_IMPOSVAL, "attempt to merge failed");
			}
		}
	| ID_MERGE any_id '(' INT ')' IDENT any_id '(' INT ')' {
			if (!almost_equals($6, "w$ith"))
				lex_error();
			id1 = which_identifier($2);
			id2 = which_identifier($7);
			col1 = $4;
			col2 = $9;
			dpp = get_data();
			if (id1 < id2) { /* swap id and col */
				id = id1; id1 = id2; id2 = id;
				id = col1; col1 = col2; col2 = id;
			}
			if (push_to_merge_table(dpp[id1], id2, col1, col2)) {
				lex_error();
				ErrMsg(ER_IMPOSVAL, "attempt to merge failed");
			}
		}
	;

bounds_cmd: ID_BOUNDS ':' QSTR { boundary_file = $3; }
	| ID_BOUNDS ':' bounds_exp
	;

bounds_exp: val { push_bound($1); }
	| bounds_exp val { push_bound($2); }
	| bounds_exp ',' val { push_bound($3); }
	;

marginals_cmd : ID_MARGINALS ':' marginals_cont

marginals_cont: val             { push_marginal(NULL, $1); }
	| marginals_cont ',' val    { push_marginal(NULL, $3); }
	| QSTR                      { push_marginal($1, -1.0); }
	| marginals_cont ',' QSTR   { push_marginal($3, -1.0); }
	;

%%

static void push_to_v(const char *mod, double sill, double range, double *d,
		int fit_sill, int fit_range) {
	VGM_MODEL vm;

	init_variogram_part(&vm);
	vm.model = which_variogram_model(mod);
	vm.range = range;
	vm.sill = sill;
	vm.fit_sill = fit_sill;
	vm.fit_range = fit_range;
	if (d != NULL && d[0] != -9999.0)
		vm.tm_range = get_tm(d);
	push_variogram_model(v, vm);
}

static int is_data_expr(DATA *d, GSTAT_EXPR *expr, const char *fld) {
#define TABLE_SIZE 27
	GSTAT_EXPR data_options[TABLE_SIZE];
	int i = 0;
#define fill_table(n, p, w, l) \
 data_options[i].name = n; data_options[i].ptr = p; \
 data_options[i].what = w; data_options[i].limit = l; i++;

/* set up table: */
	fill_table("x",          &(d->colnx),        1, 2 ) // IS_INT, GTZERO  )
	fill_table("y",          &(d->colny),        1, 2 ) // IS_INT, GTZERO  )
	fill_table("z",          &(d->colnz),        1, 2 ) // IS_INT, GTZERO  )
	fill_table("v",          &(d->colnvalue),    1, 2 ) // IS_INT, GTZERO  )
	fill_table("V",          &(d->colnvariance), 1, 2 ) // IS_INT, GTZERO  )
	fill_table("d",          &(d->polynomial_degree), 1, 2 ) // IS_INT, GEZERO  )
	fill_table("max",        &(d->sel_max),      1, 1 ) // IS_INT, GEZERO  )
	fill_table("omax",       &(d->oct_max),      1, 1 ) // IS_INT, GEZERO  )
	fill_table("min",        &(d->sel_min),      1, 1 ) // IS_INT, GEZERO  )
	fill_table("n$max",      &(d->init_max),     1, 1 ) // IS_INT, GEZERO  )
	fill_table("togrid",     &(d->togrid),       1, 1 ) // IS_INT,  GEZERO )
	fill_table("I",          &(d->Icutoff),      2, 0 ) // IS_REAL, NOLIMIT )
	fill_table("mv",         &(d->mv),           2, 0 ) // IS_REAL, NOLIMIT )
	fill_table("rad$ius",    &(d->sel_rad),      2, 2 ) // IS_REAL, GTZERO  )
	fill_table("dX",         &(d->dX),           2, 1 ) // IS_REAL, GEZERO  )
	fill_table("b$eta",      &(d->beta),         5, 0 ) // IS_D_VECTOR, NOLIMIT )
	fill_table("stan$dard",  &(d->standard),     6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("log",        &(d->log),          6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("av$erage",   &(d->average),      6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("du$mmy",     &(d->dummy),        6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("res$idual",  &(d->calc_residuals), 6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("vdist",      &(d->vdist),        6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("force",      &(d->force),        6, 0 ) // NO_ARG, NOLIMIT )
	fill_table("Cat$egory",   &(d->Category),     4, 0 ) // IS_STRING, NOLIMIT )
	fill_table("ID",         &(d->coln_id),      1, 2 ) // IS_INT, GTZERO  )
	fill_table("VarF$unction", &(d->var_fn_str),  4, 0 ) // IS_STRING, NOLIMIT  )
	fill_table(NULL, NULL, 0, 0 )

/* check TABLE_SIZE was set correctly... */
	assert(i == TABLE_SIZE);

	expr->ptr = NULL;
	expr->what = 0;//UNKNOWN;
	expr->limit = 0;//NOLIMIT;

	for (i = 0; data_options[i].name != NULL; i++) {
		if (almost_equals(fld, data_options[i].name)) {
			expr->name = fld;
			expr->ptr = data_options[i].ptr;
			expr->what = data_options[i].what;
			expr->limit = data_options[i].limit;
			if (expr->what == NO_ARG)
				*((int *) expr->ptr) = 1;
			return 1;
		}
	}

	/* non-standard cases not in data_options[] table: */
	if (almost_equals(fld, "s$tratum")) {
		if (d->id != ID_OF_VALDATA)
			return 0;
		expr->ptr = &(d->colns);
		expr->what = 1;//IS_INT;
		expr->limit = 3;//GTZERO;
		d->what_is_u = 3;//U_ISSTRATUM;
	} else if (almost_equals(fld, "av$erage")) {
		d->average = 1; expr->what = 6;//NO_ARG;
	} else if (almost_equals(fld, "noav$erage")) {
		d->average = 0; expr->what = 6;//NO_ARG;
	} else if (almost_equals(fld, "nores$idual")) {
		d->calc_residuals = 0; expr->what = 6;//NO_ARG;
	} else if (almost_equals(fld, "square")) {
		d->square = 1; expr->what = 6//NO_ARG;
	} else if (almost_equals(fld, "c")) {
		pr_warning("use `v' instead of `c' in data definition");
	} else if (almost_equals(fld, "sk_mean")) { /* move it to beta: */
		d->beta = NULL;
		d->beta = push_to_vector(-9999.0, d->beta);
		expr->ptr = &(d->beta->val[0]);
		expr->what = 3;//IS_REAL;
		expr->limit = 0;//NOLIMIT;
	}

	return (expr->what != 0);//UNKNOWN);
}

static int is_set_expr(GSTAT_EXPR *expr, const char *name) {
/*
 * parse sequences like `set zmap = 50.0;' or `set zmap = 50, idp = 2.5;'
 * (int, float or string)
 */
	int i;

	const GSTAT_EXPR set_options[] = {
	{ "cn$_max",        &gl_cn_max,       3, 2 }, // IS_REAL, GTZERO  },
	{ "co$incide",      &gl_coincide,     1, 1 }, // IS_INT,  GEZERO  },
	{ "Cr$essie",       &gl_cressie,      1, 1 }, // IS_INT,  GEZERO  },
	{ "a$lpha",         &gl_alpha,        3, 1 }, // IS_REAL, GEZERO  },
	{ "b$eta",          &gl_beta,         3, 1 }, // IS_REAL, GEZERO  },
	{ "c$utoff",        &gl_cutoff,       3, 2 }, // IS_REAL, GTZERO  },
	{ "de$bug",         &debug_level,     1, 1 }, // IS_INT,  GEZERO  },
	{ "display",        &gl_display,      4, 0 }, // IS_STRING, NOLIMIT },
	{ "do$ts",          &gl_dots,         1, 1 }, // IS_INT,  GEZERO  },
	{ "fit",            &gl_fit,          1, 1 }, // IS_INT,  GEZERO  },
	{ "fit_l$imit",     &gl_fit_limit,    3, 2 }, // IS_REAL, GTZERO  },
	{ "fo$rmat",        &gl_format,       4, 0 }, // IS_STRING, NOLIMIT },
	{ "fr$action",      &gl_fraction,     3, 2 }, // IS_REAL, GTZERO  },
	{ "gcv",            &gl_gcv,          3, 2 }, // IS_REAL, GTZERO  },
	{ "gls$_residuals", &gl_gls_residuals, 1, 1 }, // IS_INT, GEZERO  },
	{ "gnuplot",        &gl_gnuplot,      4, 0 }, // IS_STRING, NOLIMIT },
	{ "gnuplot35",      &gl_gnuplot35,    4, 0 }, // IS_STRING, NOLIMIT },
	{ "gpt$erm",        &gl_gpterm,       4, 0 }, // IS_STRING, NOLIMIT  },
	{ "id$p",           &gl_idp,          3, 2 }, // IS_REAL, GTZERO  },
	{ "in$tervals",     &gl_n_intervals,  1, 2 }, // IS_INT,  GTZERO  },
	{ "it$er",          &gl_iter,         1, 1 }, // IS_INT,  GEZERO  },
	{ "j$graph",        &gl_jgraph,       1, 1 }, // IS_INT,  GEZERO  },
	{ "lhs",            &gl_lhs,          1, 1 }, // IS_INT,  GEZERO  },
	{ "log$file",       &logfile_name,    4, 0 }, // IS_STRING, NOLIMIT },
	{ "mv$string",		&gl_mv_string,    4, 0 }, // IS_STRING, NOLIMIT },
	{ "n_uk",           &gl_n_uk,         1, 1 }, // IS_INT,  GEZERO  },
	{ "numbers",        &gl_numbers,      1, 1 }, // IS_INT,  GEZERO  },
	{ "nb$lockdiscr",   &gl_nblockdiscr,  1, 2 }, // IS_INT,  GTZERO  },
	{ "no$check",       &gl_nocheck,      1, 1 }, // IS_INT,  GEZERO  },
	{ "ns$im",          &gl_nsim,         1, 2 }, // IS_INT,  GTZERO  },
	{ "o$utputfile",    &o_filename,      4, 0 }, // IS_STRING, NOLIMIT },
	{ "or$der",         &gl_order,        1, 1 }, // IS_INT,  GEZERO },
	{ "pag$er",         &gl_pager,        4, 0 }, // IS_STRING, NOLIMIT },
	{ "pl$otfile",      &gl_gnufile,      4, 0 }, // IS_STRING, NOLIMIT },
	{ "q$uantile",      &gl_quantile,     3, 2 }, // IS_REAL, GTZERO  },
	{ "rp",             &gl_rp,           1, 1 }, // IS_INT,  GEZERO  },
	{ "sec$ure",        &gl_secure,       1, 2 }, // IS_INT,  GTZERO  },
	{ "see$d",          &gl_seed,         1, 2 }, // IS_INT,  GTZERO  },
	{ "useed",          &gl_seed,         2, 1 }, // IS_UINT,  GEZERO  },
	{ "spa$rse",        &gl_sparse,       1, 1 }, // IS_INT,  GEZERO  },
	{ "spi$ral",        &gl_spiral,       1, 1 }, // IS_INT,  GEZERO  },
	{ "spl$it",         &gl_split,        1, 2 }, // IS_INT,  GTZERO  },
	{ "sy$mmetric",     &gl_sym_ev,       1, 1 }, // IS_INT,  GEZERO  },
	{ "tol_h$or",       &gl_tol_hor,      3, 1 }, // IS_REAL, GEZERO  },
	{ "tol_v$er",       &gl_tol_ver,      3, 1 }, // IS_REAL, GEZERO  },
	{ "v$erbose",       &debug_level,     1, 1 }, // IS_INT,  GEZERO  },
	{ "w$idth",         &gl_iwidth,       3, 1 }, // IS_REAL, GEZERO  },
	{ "x$valid",        &gl_xvalid,       1, 1 }, // IS_INT,  GEZERO  },
	{ "zero_di$st",     &gl_zero_est,     1, 1 }, // IS_INT,  GEZERO  },
	{ "zero",           &gl_zero,         3, 1 }, // IS_REAL, GEZERO  },
	{ "zm$ap",          &gl_zmap,         3, 0 }, // IS_REAL, NOLIMIT },
	{ NULL, NULL, 0, 0 }
	};

	for (i = 0; set_options[i].name; i++)
		if (almost_equals(name, set_options[i].name))
			break; /* break out i-loop */
	if (set_options[i].name == NULL)
		return 0;

	if (almost_equals((const char *)name,"nb$lockdiscr"))
		gl_gauss = 0; /* side effect */

	expr->name = name;
	expr->ptr = set_options[i].ptr;
	expr->what = set_options[i].what;
	expr->limit = set_options[i].limit;

	return 1;
}

static void check_assign_expr(GSTAT_EXPR *expr) {
/* for INT and REAL expressions, check range */
	double val;

	switch(expr->what) {
		case 1://IS_INT:
			val = (double) (*((int *)(expr->ptr)));
			break;
		case 3://IS_REAL:
			val = (*((double *)(expr->ptr)));
			break;
		default:
			return;
	}
	if (expr->limit == 1/*GEZERO*/ && val < 0.0) {
		lex_error();
		pr_warning("value should be non-negative");
		ErrMsg(ER_IMPOSVAL, expr->name);
	}
	if (expr->limit == 2/*GTZERO*/ && val <= 0.0) {
		lex_error();
		pr_warning("value should be positive");
		ErrMsg(ER_IMPOSVAL, expr->name);
	}
}

static void push_data_X(DATA *d, int id) {
	if (id == -1) { /* remove default intercept */
		if (d->n_X > 1) {
			lex_error();
			ErrMsg(ER_SYNTAX, "-1 only as first argument following X=");
		}
		d->n_X = 0;
	} else if (id == 0) {
		lex_error();
		ErrMsg(ER_SYNTAX, "intercept is default");
	} else /* id > 0 */
		data_add_X(d, id);
}

static int is_block_expr(GSTAT_EXPR *expr, const char *s) {
	DPOINT *bp;

	bp = get_block_p();
	expr->name = s;
	expr->limit = 1;//GEZERO;
	expr->what = 3;//IS_REAL;
	if (almost_equals(s, "dx"))
		expr->ptr = &(bp->x);
	else if (almost_equals(s, "dy"))
		expr->ptr = &(bp->y);
	else if (almost_equals(s, "dz"))
		expr->ptr = &(bp->z);
	else
		return 0;
	return 1;
}

static ANIS_TM *get_tm(double anis[5]) {
/* Part of this routine was taken from GSLIB, first edition:
C%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
C                                                                      %
C Copyright (C) 1992 Stanford Center for Reservoir Forecasting.  All   %
C rights reserved.  Distributed with: C.V. Deutsch and A.G. Journel.   %
C ``GSLIB: Geostatistical Software Library and User's Guide,'' Oxford  %
C University Press, New York, 1992.                                    %
C                                                                      %
C The programs in GSLIB are distributed in the hope that they will be  %
C useful, but WITHOUT ANY WARRANTY.  No author or distributor accepts  %
C responsibility to anyone for the consequences of using them or for   %
C whether they serve any particular purpose or work at all, unless he  %
C says so in writing.  Everyone is granted permission to copy, modify  %
C and redistribute the programs in GSLIB, but only under the condition %
C that this notice and the above copyright notice remain intact.       %
C                                                                      %
C%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
*/
	int i;
	double alpha, beta, theta, sina, sinb, sint, cosa, cosb, cost, afac1, afac2;
	ANIS_TM *t = NULL;

/*
	About naming convention:

	gstat     GSLIB
	===============
	anis[0]    ang1 (first anis. par. for 2D)
	anis[1]    ang2
	anis[2]    ang3
	anis[3]   anis1 (second anis. par. for 2D)
	anis[4]   anis2
*/

#define ANIS_ERR(x) message("parsing anis. pars. %g,%g,%g,%g,%g -- error on %g\n", \
	anis[0],anis[1],anis[2],anis[3],anis[4],x)
#define DEG2RAD (PI/180.0)

	for (i = 0; i < 3; i++) {
		if (anis[i] < 0 || anis[i] >= 360) {
			ANIS_ERR(anis[i]);
			ErrMsg(ER_RANGE, "this value should be in [0..360>");
		}
	}
	for (i = 3; i < 5; i++) {
		if (anis[i] <= 0.0 || anis[i] > 1.0) {
			ANIS_ERR(anis[i]);
			ErrMsg(ER_RANGE, "this value should be in <0..1]");
		}
	}

	/* from GSLIB: */
	if (anis[0] >= 0.0 && anis[0] < 270)
		alpha = (double) (90.0 - anis[0]) * DEG2RAD;
	else
		alpha = (double) (450.0 - anis[0]) * DEG2RAD;
	beta = -1.0 * (double) anis[1] * DEG2RAD;
	theta =       (double) anis[2] * DEG2RAD;

	sina = sin(alpha);
	sinb = sin(beta);
	sint = sin(theta);
	cosa = cos(alpha);
	cosb = cos(beta);
	cost = cos(theta);

	afac1 = 1.0 / MAX((double) anis[3], (double) EPSILON);
	afac2 = 1.0 / MAX((double) anis[4], (double) EPSILON);

	t = (ANIS_TM *) emalloc(sizeof(ANIS_TM));

	t->angle[0] = anis[0];
	t->angle[1] = anis[1];
	t->angle[2] = anis[2];
	t->ratio[0] = anis[3];
	t->ratio[1] = anis[4];
	t->tm[0][0] =       (cosb * cosa);
	t->tm[0][1] =       (cosb * sina);
	t->tm[0][2] =       (-sinb);
	t->tm[1][0] = afac1*(-cost*sina + sint*sinb*cosa);
	t->tm[1][1] = afac1*(cost*cosa + sint*sinb*sina);
	t->tm[1][2] = afac1*( sint * cosb);
	t->tm[2][0] = afac2*(sint*sina + cost*sinb*cosa);
	t->tm[2][1] = afac2*(-sint*cosa + cost*sinb*sina);
	t->tm[2][2] = afac2*(cost * cosb);
	return t;
}

static void push_marginal(char *name, double value) {
	static int names = -1;

	if ((names == -1) && (name != NULL))
		names = *(name);

	if (name) {
		if (!names) {
			lex_error();
			ErrMsg(ER_SYNTAX, "only real values allowed"); 
		}
		gl_marginal_names = (char **) erealloc(gl_marginal_names,
			++gl_n_marginals * sizeof(char *));
		gl_marginal_names[gl_n_marginals - 1] = name;
	} else {
		if (names) {
			lex_error();
			ErrMsg(ER_SYNTAX, "only quoted strings allowed"); 
		}
		gl_marginal_values = (double *) erealloc (gl_marginal_values,
			++gl_n_marginals * sizeof(double));
		gl_marginal_values[gl_n_marginals - 1] = value;
	}
	return;
}

static void reset_parser(void) {
/* savety first: reset all static globals (should be unnessesary) */
	v = NULL;
	d = NULL;
	bp = NULL;
	ofn = NULL;
	expr.ptr = NULL;
	expr.what =  0;//UNKNOWN;
	expr.limit =  0;//NOLIMIT;
	id = id1 = id2 = col1 = col2 = -1;
}

int parse_cmdo(const char *cmd, const char *fname) {
	set_lex_source(const_cast <char *>(cmd), fname);
	reset_parser();
	return yyparse();
}

int parse_file(const char *fname) {
/* 
 * parse commands in file fname
 */
	int stdin_isatty = 1;
	char *cp;

	if (fname == NULL || strcmp(fname, "-") == 0) {
#ifdef HAVE_UNISTD_H
		stdin_isatty = isatty(fileno(stdin));
#endif
		if (stdin_isatty)
			cp = string_prompt("gstat> ");
		else
			cp = string_file(NULL);
	} else /* read from file */
		cp = string_file(fname);

	if (parse_cmd(cp, fname))
		ErrMsg(ER_SYNTAX, fname);
	efree(cp);

	if (boundary_file != NULL) {
		cp = string_file(boundary_file);
		if (parse_cmd(cp, boundary_file))
			ErrMsg(ER_SYNTAX, boundary_file);
		efree(cp);
	}

	if (vector_only && !allow_vector_only)
		ErrMsg(ER_SYNTAX, fname);

	return 0;
}

 void parse_gstatrc(void) {
	char *fname = NULL, *cp;

	if ((fname = getenv("GSTATRC")) != NULL) {
		if (! file_exists(fname)) {
			message("environment variable %s:\n", GSTATRC);
			ErrMsg(ER_READ, fname);
		}
		parse_file(fname);
	} else if ((cp = getenv("HOME")) != NULL) {
		fname = (char *) emalloc(strlen(cp) + strlen(HOMERCFILE) + 2);
		sprintf(fname, "%s/%s", cp, HOMERCFILE);
		if (file_exists(fname))
			parse_file(fname);
		efree(fname);
	}
	return;
}

int read_variogram(VARIOGRAM *v, const char *source) {
	char *cp;
	int rval;

	parse_variogram = v;
	cp = (char *) emalloc((strlen(source) + 20) * sizeof(char));
	sprintf(cp, "variogram(): %s;", source);
	rval = parse_cmd(cp, NULL);
	parse_variogram = NULL; /* for savety */
	efree(cp);
	return rval;
}

int read_vector(D_VECTOR *d, char *fname) {
	int rval;

	assert(d != NULL);
	sd_vector = d;

	allow_vector_only = 1;

	rval = parse_file(fname);

	if (! vector_only)  {
		message("stat: only numeric input allowed -- \n");
		ErrMsg(ER_IMPOSVAL, fname);
	}

	return rval;
}

static void verify_data(DATA *d) { /* declaration : contents */

	if (d->var_fn_str != NULL) {
		if (almost_equals(d->var_fn_str, "mu")) {
			d->variance_fn = v_mu;
		} else if (almost_equals(d->var_fn_str, "mu(1-mu)")) {
			d->variance_fn = v_bin;
		} else {
			lex_error();
			message("variance function not supported:\n");
			ErrMsg(ER_SYNTAX, d->var_fn_str);
		}
	}
}
