/*
    Gstat, a program for geostatistical modelling, prediction and simulation
    Copyright 1992, 1999 (C) Edzer J. Pebesma

    Edzer J. Pebesma, e.pebesma@geog.uu.nl
    Department of physical geography, Utrecht University
    P.O. Box 80.115, 3508 TC Utrecht, The Netherlands

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (read also the files COPYING and Copyright)
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "defs.h"
/*#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifndef HAVE_GETOPT*/
# include "getopt.h"
/*#endif*/

#include "utils.h"
#include "mapio.h"
#include "userio.h"

/*
External (wrt. gstat) uses:
1. gd 1.2 gif library, by Thomas Boutell, www.boutell.com.
   Get the libraray from http://www.boutell.com/gd/gd1.2.tar.Z,
   Read the API local or at http://www.boutell.com/gd/
   The makefile assumes that gd 1.2 library and include files
   reside in ../gd1.2
2. gifmerge, Rev 1.33 (C) 1991,1992 by Mark Podlipec, Improvements by
   Rene K. Mueller 1996 (http://www.iis.ee.ethz.ch/~kiwi/GIFMerge/)
   for animated gifs (second usage). Assumes gifmerge is in the search
   path.
*/

#define OPTIONS "\
-c f  colour table file f (%r %g %b on each line)\n\
-i    make interlaced gif\n\
-m $  set minimum value to $\n\
-M $  set maximum value to $\n\
-t    no transparant background\n\
-T s  use string s as legend title (-T """" to suppress title)\n\
-p n  use n pixels in gif per map grid cell\n\
-l n  print legend, n: 0 no legend; 1 right; 2 bottom; 3 classes\n\
-n    nice automatic legend boundaries\n\
-s    silent mode\n\
-C 1,2,3,4 use class boundaries 1,2,3,4\n\
"

#define LEGSIZEX 90
#define LEGSIZEY 45

#ifdef HAVE_LIBCSF
# include "/gstat-2.1.0/include/csf.h"
# define IS_CLASSIFIED(m) (legend == 3 || \
	(m->CSF_MAP && RgetValueScale(m->CSF_MAP) != VS_SCALAR && \
	RgetValueScale(m->CSF_MAP) != VS_CONTINUOUS))
#else
 #define IS_CLASSIFIED(m) (legend == 3)
#endif
#define COLOUR_TSIZE 16

int px = 0, interlace = 0, transparent = 1, legend = 0, silent = 0,
	min_set = 0, max_set = 0, table_read = 0, nc = 0, nice_legend = 0,
	nc_max = 0, n_table = 0, cont_table = 0;
float /*min, max,*/ interval, *table; /*KS removed min,max*/
char name[256] = "", *title = NULL, *gifmerge = "gifmerge -255,255,255 -l0";

#ifdef STANDALONE
# define map2gif main
#endif

#ifndef HAVE_LIBGD
int map2gif(int argc, char *argv[]) {
	printlog("for map2gif, install the gd gif library (http://www.boutell.com/gd/)\n");
	printlog("then, use  ./configure --with-gd; make; make install\n");
	ErrMsg(ER_ARGOPT, "map2gif");
	return 0;
}
#else

#include "gd.h"

static int do_one_map2gif(char *map, const char *gif);
static void draw_one_cell(gdImagePtr g, int x, int y, int pixx, int pixy, 
		int color);
static double find_reasonable_hash_start(double min, double max,
		double hash_interval);
static double find_reasonable_hash_interval(double range);
static void read_table(char *s);
static int find_entry(float *table, int n_table, float value);

/* 
 * convert grid map to gif image 
 */
int map2gif(int argc, char *argv[]) {
	int c, i, setup_title;
	char *cmd = NULL, *tmp = NULL, **names = NULL;
	extern char *optarg;
	extern int optind, opterr, optopt;

	opterr = 0;
	while ((c = getopt(argc, argv, "c:C:il:m:M:np:stT:x")) != EOF) {
		switch (c) {
			case 'c': sprintf(name, "%s", optarg); break;
			case 'i': interlace = 1; break;
			case 'l': legend = atoi(optarg); break;
			case 'm': min = atof(optarg); min_set = 1; break;
			case 'M': max = atof(optarg); max_set = 1; break;
			case 'n': nice_legend = 1; break;
			case 'p': px = atoi(optarg); break;
			case 's': silent = 1; break;
			case 'C': read_table(optarg); break;
			case 't': transparent = 0; break;
			case 'T': title = optarg; break;
			case 'x': cont_table = 1; break;
			case '?': default: ErrClo(optopt);
		}
	}

	if (argc - optind < 2) {
		printf("map2gif: Copyright 1997 Edzer J. Pebesma\n");
		printf("usage: map2gif [options] map gif_file\n");
		printf("usage: map2gif [options] anim_gif map1 map2 ... (uses gifmerge)\n");
		printf("options:\n%s", OPTIONS);
		return 0;
	}

	if (! silent) {
		printf("uses the gd gif library by Thomas Boutell, http://www.boutell.com/gd/\n");
		printf("gd 1.2 Copyright 1994, 1995,\n\tQuest Protein Database Center Cold Spring Harbor Labs\n");
	}

	if (argc - optind == 2) 
		return do_one_map2gif(argv[optind], argv[optind+1]);

	setup_title = (title == NULL); /* not set with -T */
	for (i = optind, c = strlen(gifmerge) + 100; i < argc; i++)
		c += strlen(argv[i]) + 1;
	cmd = emalloc(c);
	names = (char **) emalloc(argc);
	sprintf(cmd, "%s ", gifmerge);
	for (i = optind + 1; i < argc; i++) {
		tmp = tmpnam(NULL);
		names[i] = string_dup(tmp);
		if (setup_title)
			title = argv[i];
		do_one_map2gif(argv[i], tmp);
		strcat(cmd, tmp);
		strcat(cmd, " ");
	}
	strcat(cmd, "> ");
	strcat(cmd, argv[optind]);
	printf("executing [%s] ...\n", cmd);
	if ((i = esystem(cmd)))
		printf("%s returned exit code %d\n", gifmerge, i);
	for (i = optind + 1; i < argc; i++) 
		remove(names[i]);
	return 0;
}


static int do_one_map2gif(char *map, const char *gif) {
	FILE *out = NULL;
	GRIDMAP *m = NULL;
#ifdef HAVE_LIBCSF
	CSF_LEGEND *l = NULL;
#else
	typedef unsigned int UINT2;
#endif
	gdImagePtr g;
	unsigned int i, j, class, sx, sy;
	int gcol[256], black, si;
	float value, R, G, B;
	extern gdFontPtr gdFontSmall; 
	gdFontPtr font;
	static UINT2 *col_buf = NULL, scalar_table[3 * COLOUR_TSIZE] = {
		25, 127, 0,
		51, 153, 0,
		89, 173, 0,
		127, 191, 0,
		153, 204, 0,
		178, 216, 0,
		216, 237, 0,
		242, 255, 0,
		250, 214, 0,
		255, 191, 0,
		255, 168, 0,
		255, 127, 0,
		255, 102, 0,
		255, 84, 0,
		255, 51, 0,
		255, 0, 0 },
	nominal_table[3 * COLOUR_TSIZE] = {
		0,128,128,
		255,128,0,
		128,128,0,
		0,128,0,
		0,128,255,
		128,0,0,
		128,0,255,
		0,0,128,
		128,0,128,
		255,0,128,
		0,188,188,
		255,188,0,
		188,188,0,
		0,188,0,
		0,188,255,
		188,0,0 };
		
	font = gdFontSmall;
	if (!title /*== NULL*/)
		title = map; /* suppress this with -T "" */
	m = new_map();
	m->filename = map;
	if (NULL == (m = map_read(m)))
		ErrMsg(ER_READ, map);
	if (px < 1)
		px = MAX(1, (500 / MAX(m->cols, m->rows))); /* default max to 500 px */
	if (! table_read) {
#ifdef HAVE_LIBCSF
		if (m->CSF_MAP /*!= NULL*/) { /* try to read table from map */
			if ((nc = MgetNrColourPaletteEntries(m->CSF_MAP)) > 0) {
			/* colour table is present in map: */
				col_buf = emalloc(nc * 3 * sizeof(UINT2));
				MgetColourPalette(m->CSF_MAP, col_buf);
			} else if ((nc = MgetNrGreyPaletteEntries(m->CSF_MAP)) > 0) {
				col_buf = emalloc(nc * 3 * sizeof(UINT2));
				MgetGreyPalette(m->CSF_MAP, col_buf);
				for (si = nc - 1; si >= 0; si--) { /* fill colours backwards */
					col_buf[3 * si + 2] = col_buf[si];
					col_buf[3 * si + 1] = col_buf[si];
					col_buf[3 * si] = col_buf[si];
				}
			}
			if ((i = MgetNrLegendEntries(m->CSF_MAP)) > 0) {
				l = emalloc(i * sizeof(CSF_LEGEND));
				MgetLegend(m->CSF_MAP, l);
				if (legend > 0)
					legend = 3;
			}
			for (i = 0; i < nc; i++) {
				col_buf[i*3] = col_buf[i*3] >> 8; /* MSB */
				col_buf[i*3+1] = col_buf[i*3+1] >> 8;
				col_buf[i*3+2] = col_buf[i*3+2] >> 8;
			}
		}
#endif
		if (nc == 0) { /* still... */
			if (name[0] == '\0' && getenv("PCR_DIR") != NULL)
				sprintf(name, "%s%s", getenv("PCR_DIR"),
					IS_CLASSIFIED(m) ? "class.pal" : "con.pal");
			if (NULL != (out = fopen(name, "r"))) { 
				/* found colour table file -- read it */
				if (! silent)
					printf("reading colour table from `%s'", name);
				col_buf = emalloc(256 * 3 * sizeof(UINT2));
				i = 0;
				while (fscanf(out, "%g %g %g", &R, &G, &B) == 3) {
					col_buf[i * 3]     = (UINT2) floor((R * 2.55) + 0.5);
					col_buf[i * 3 + 1] = (UINT2) floor((G * 2.55) + 0.5);
					col_buf[i * 3 + 2] = (UINT2) floor((B * 2.55) + 0.5);
					i++;
				}
				fclose(out);
				nc = i;
				if (! silent)
					printf(", %d colours\n", nc);
			} else { /* use default, built-in colour tables */
				col_buf = (IS_CLASSIFIED(m) && !cont_table) ? 
					nominal_table : scalar_table;
				nc = COLOUR_TSIZE;
				if (! silent)
					printf("using internal colour table, %d colours\n", nc);
			}
		}
		if (! silent) {
			printf("transparent is 255,255,255\n");
			for (i = 0; i < nc; i++)
				printf("colour %d: %u %u %u\n", i,
					col_buf[i * 3], col_buf[i * 3 + 1], col_buf[i * 3 + 2]);
		}
		table_read = 1;
	}

	if (n_table > 0) {
		min = table[0];
		max = table[n_table-1];
		min_set = max_set = 1;
		nc = n_table - 1;
	}

	if (nice_legend)
		interval = find_reasonable_hash_interval(m->cellmax - m->cellmin);
	
	if (! min_set) {
		if (nice_legend)
			min = find_reasonable_hash_start(m->cellmin, m->cellmax, interval);
		else
			min = m->cellmin;
	}
	if (! max_set) {
		if (nice_legend) {
			max = min;
			nc = 0;
			while (max < m->cellmax) {
				max += interval;
				nc++;
			}
		} else
			max = m->cellmax;
	}

	printf("min: %g, max: %g, nc: %d\n", min, max, nc);

	nc_max = nc;

	if (legend == 3 && max - min < nc)
		nc = ceil(max - min) + 1;

	if (nc > 254) {
		printf("too many colours in %s (max 254)\n", map);
		exit(1);
	}

	sx = m->cols * px + (legend && legend != 2) * LEGSIZEX;
	sy = m->rows * px + (legend == 2) * LEGSIZEY;

	g = gdImageCreate(sx, sy);

	gcol[nc] = gdImageColorAllocate(g, 255, 255, 255); /* back ground: white */
	black = gdImageColorAllocate(g, 0, 0, 0); /* text colour */
	if (legend) {
		for (i = 0; i < nc; i++) {
			if (legend == 3)
				class = i + min;
			else
				class = i;
			assert(class <= nc_max);
			gcol[i] = gdImageColorAllocate(g, col_buf[class * 3],
					col_buf[class * 3 + 1], col_buf[class * 3 + 2]);
		}
	} else {
		for (i = 0; i < nc; i++)
			gcol[i] = gdImageColorAllocate(g, col_buf[i * 3],
					col_buf[i * 3 + 1], col_buf[i * 3 + 2]);
	}

	if (transparent)
		gdImageColorTransparent(g, gcol[nc]); /* white -> transparent */

	gdImageInterlace(g, interlace);

	draw_one_cell(g, 0, 0, sx, sy, gcol[nc]); 
	/* initialize gif white/transparent */

	/* draw cells: */
	for (i = 0; i < m->rows; i++) {
		for (j = 0; j < m->cols; j++) {
			if (! map_cell_is_mv(m, i, j)) {
				value = map_get_cell(m, i, j) - min; /* [0,...,max-min] */
				if (IS_CLASSIFIED(m)) {
					/* map as integers to [0,...,nc>: */
					class = floor(value);
					if (nc > 1 && class >= nc) /* OVERFLOW------>>> */
						class = class % (nc - 1); /* repeat colours */
				} else {
					/* map to [0,...,nc-1]: */
					if (!table /*== NULL*/) {
						class = (max == min) ? 0 : floor(nc * value/(max-min));
						if (class == nc) /* occurs only when value == max-min */
						class--;
					} else
						class = find_entry(table, n_table, value);
				}
				if (class != -1)
					draw_one_cell(g, j * px, i * px, px, px, gcol[class]);
			} 
		}
	}
	switch (legend) {
		case 0: /* no legend: */ break;
		case 1:
			for (i = 0; i <= nc; i++) {
				j = px * m->rows / 2 + (i - nc / 2) * 10; /* half way up/down */
				if (i < nc)
					draw_one_cell(g, m->cols * px + 10, j, 20, 10, gcol[i]);
				if (table)
					sprintf(name, "%4g", table[i]);
				else
					sprintf(name, "%4g", min + i*(max-min)/nc);
				gdImageString(g, font, m->cols * px + 35, j - 8, 
						(unsigned char *) name, black);
			}

			/* min value: 
			sprintf(name, "%4g", min);
			j = px * m->rows / 2 - (nc + 1) * 5;
			gdImageString(g, font, m->cols * px + 35, j, 
					(unsigned char *) name, black);
			*/

			/* max value: */
			if (table)
				sprintf(name, "%4g", table[n_table-1]);
			else
				sprintf(name, "%4g", max);

			/*
			j = px * m->rows / 2; 
			gdImageString(g, font, m->cols * px + 35, j - 8, 
					(unsigned char *) name, black);
			*/

			/* was:
			j = px * m->rows / 2 + (nc - 1) * 5 + 3;
			gdImageString(g, font, m->cols * px + 35, j, 
					(unsigned char *) name, black);
			*/

			/* title: */
			j = px * m->rows / 2 - (nc + 4) * 5;
			gdImageString(g, font, m->cols * px + 10, j, 
					(unsigned char *) title, black);
			break;
		case 2:
			for (i = 0; i < nc; i++) {
				j = px * m->cols / 2 + (i - nc / 2) * 10; 
				draw_one_cell(g, j, m->rows * px + 10, 10, 20, gcol[i]);
			}
			sprintf(name, "%6g", min);
			j = px * m->cols / 2 - (nc + 4) * 5;
			gdImageString(g, font, j, m->rows * px + 35, 
					(unsigned char *) name, black);
			sprintf(name, "%6g", max);
			j = px * m->cols / 2 + (nc - 4) * 5;
			gdImageString(g, font, j, m->rows * px + 35, 
					(unsigned char *) name, black);
			j = px * m->cols / 2 + (1 + nc / 2) * 10; /* right */
			gdImageString(g, font, j, m->rows * px + 20, 
					(unsigned char *) title, black);
			break;
		case 3:
			for (class = min; class <= max; class++) {
				i = class - min; /* [0...nc> */

				assert(i < nc);

				j = px * m->rows / 2 + (i - nc / 2) * 15; /* half way up/down */
				draw_one_cell(g, m->cols * px + 10, j, 20, 10, gcol[i]);
#ifdef HAVE_LIBCSF
				if (MgetNrLegendEntries(m->CSF_MAP))
					assert(class < MgetNrLegendEntries(m->CSF_MAP));

				if (l)
					sprintf(name, "%s", l[class].descr);
				else
#endif
					sprintf(name, "%3d", (int) floor(min + i));
				gdImageString(g, font, m->cols * px + 35, j, 
						(unsigned char *) name, black);
			}
			j = px * m->rows / 2 + (-1 - nc / 2) * 15; /* above */
			gdImageString(g, font, m->cols * px + 10, j, 
					(unsigned char *) title, black);
			break;
		default:
			printf("unknown legend option, %d\n", legend);
			exit(1);
	} /* switch legend */

	/* write output to gif */
	out = fopen(gif, "wb");
#ifdef HAVE_GDIMAGEGIF
	gdImageGif(g, out);
#else
# ifdef HAVE_GDIMAGEPNG /* libgd 1.6.1+ */
	gdImagePng(g, out);
# else
#  error gdImageGif and gdImagePng are both unavailable
# endif
#endif
	fclose(out);

	/* clean up: */
	gdImageDestroy(g);
	map_free(m);

	return 0;
}

void draw_one_cell(gdImagePtr g, int x, int y, int pixx, int pixy, int color) {
	int i, j;
	for (i = 0; i < pixx; i++)
		for (j = 0; j < pixy; j++)
			gdImageSetPixel(g, x+i, y+j, color);
}

static double find_reasonable_hash_interval(double range) {
  double d = 1.0;

  /* range = max - min; */
  if (range > 5.0) {
    while(1) {
      if (range / d < 6.0) return d;
      d *= 2.0;
      if (range / d < 6.0) return d;
      d *= 2.5;
      if (range / d < 6.0) return d;
      d *= 2.0;
    }
  } else {
    while(1) {
      if (range / d > 2.0) return d;
      d /= 2.0;
      if (range / d > 2.0) return d;
      d /= 2.5;
      if (range / d > 2.0) return d;
      d /= 2.0;
    }
  }
}

static double find_reasonable_hash_start(double min, double max, double hash_interval) {
  int i;
  double d = 0.0;
  
  if (max > 0.0 && min < 0.0) {
  	/* return 0.0; */
  	while (d > min)
  		d -= hash_interval;
  	return d;
  }
  i = ((int) (min / hash_interval));
  return ((double) i) * hash_interval;
}

static void read_table(char *s) {
	char *dup, *token = NULL;

	dup = string_dup(s); /* mess this one up */
	table = (float *) emalloc(1000 * sizeof(float));
	printf("table:\n");
	while (NULL != (token = strtok(n_table == 0 ? dup : NULL, ", ")) /*!= NULL*/) {
		table[n_table] = atof(token);
		printf("%f ", table[n_table]);
		n_table++;
		assert(n_table < 1000);
	}
	printf("\n");
}

static int find_entry(float *table, int n_table, float value) {
	int i = 0;

	if (value < table[0])
		return -1;
	while (value > table[i])
		i++;
	if (i == n_table)
		return -1;
	return i - 1;
}

#endif /* #ifdef HAVE_LIBGD */
